package org.xbib.helianthus.client;

import io.netty.channel.EventLoop;
import org.xbib.helianthus.common.Request;
import org.xbib.helianthus.common.Response;
import org.xbib.helianthus.common.SessionProtocol;

import java.util.function.Function;
import java.util.function.Supplier;

public abstract class UserClient<T, I extends Request, O extends Response> implements ClientOptionDerivable<T> {

    private final Client<I, O> delegate;
    private final Supplier<EventLoop> eventLoopSupplier;
    private final SessionProtocol sessionProtocol;
    private final ClientOptions options;
    private final Endpoint endpoint;

    protected UserClient(Client<I, O> delegate, Supplier<EventLoop> eventLoopSupplier,
                         SessionProtocol sessionProtocol, ClientOptions options, Endpoint endpoint) {

        this.delegate = delegate;
        this.eventLoopSupplier = eventLoopSupplier;
        this.sessionProtocol = sessionProtocol;
        this.options = options;
        this.endpoint = endpoint;
    }

    @SuppressWarnings("unchecked")
    protected final <U extends Client<I, O>> U delegate() {
        return (U) delegate;
    }

    protected final EventLoop eventLoop() {
        return eventLoopSupplier.get();
    }

    protected final SessionProtocol sessionProtocol() {
        return sessionProtocol;
    }

    protected final ClientOptions options() {
        return options;
    }

    protected final Endpoint endpoint() {
        return endpoint;
    }

    protected final O execute(
            String method, String path, I req, Function<Throwable, O> fallback) {
        return execute(eventLoop(), method, path, req, fallback);
    }

    protected final O execute(
            EventLoop eventLoop, String method, String path, I req, Function<Throwable, O> fallback) {

        final ClientRequestContext ctx = new DefaultClientRequestContext(
                eventLoop, sessionProtocol, endpoint, method, path, options, req);
        try {
            return delegate().execute(ctx, req);
        } catch (Throwable cause) {
            ctx.responseLogBuilder().end(cause);
            return fallback.apply(cause);
        }
    }

    @Override
    public final T withOptions(ClientOptionValue<?>... additionalOptions) {
        final ClientOptions options = ClientOptions.of(options(), additionalOptions);
        return newInstance(delegate(), eventLoopSupplier, sessionProtocol(), options, endpoint());
    }

    @Override
    public final T withOptions(Iterable<ClientOptionValue<?>> additionalOptions) {
        final ClientOptions options = ClientOptions.of(options(), additionalOptions);
        return newInstance(delegate(), eventLoopSupplier, sessionProtocol(), options, endpoint());
    }

    protected abstract T newInstance(Client<I, O> delegate, Supplier<EventLoop> eventLoopSupplier,
                                     SessionProtocol sessionProtocol, ClientOptions options, Endpoint endpoint);
}
