package org.xbib.helianthus.client.dns;

import static io.netty.util.internal.ObjectUtil.checkNotNull;

import io.netty.resolver.NameResolver;
import io.netty.util.concurrent.EventExecutor;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.FutureListener;
import io.netty.util.concurrent.Promise;
import io.netty.util.internal.StringUtil;

import java.util.List;
import java.util.concurrent.ConcurrentMap;

/**
 *
 * @param <T> parameter type
 */
final class InflightNameResolver<T> implements NameResolver<T> {

    private final EventExecutor executor;
    private final NameResolver<T> delegate;
    private final ConcurrentMap<String, Promise<T>> resolvesInProgress;
    private final ConcurrentMap<String, Promise<List<T>>> resolveAllsInProgress;

    InflightNameResolver(EventExecutor executor, NameResolver<T> delegate,
                         ConcurrentMap<String, Promise<T>> resolvesInProgress,
                         ConcurrentMap<String, Promise<List<T>>> resolveAllsInProgress) {

        this.executor = checkNotNull(executor, "executor");
        this.delegate = checkNotNull(delegate, "delegate");
        this.resolvesInProgress = checkNotNull(resolvesInProgress, "resolvesInProgress");
        this.resolveAllsInProgress = checkNotNull(resolveAllsInProgress, "resolveAllsInProgress");
    }

    @Override
    public Future<T> resolve(String inetHost) {
        return resolve(inetHost, executor.<T>newPromise());
    }

    @Override
    public Future<List<T>> resolveAll(String inetHost) {
        return resolveAll(inetHost, executor.<List<T>>newPromise());
    }

    @Override
    public void close() {
        delegate.close();
    }

    @Override
    public Promise<T> resolve(String inetHost, Promise<T> promise) {
        return resolve(resolvesInProgress, inetHost, promise, false);
    }

    @Override
    public Promise<List<T>> resolveAll(String inetHost, Promise<List<T>> promise) {
        return resolve(resolveAllsInProgress, inetHost, promise, true);
    }

    private <U> Promise<U> resolve(
            final ConcurrentMap<String, Promise<U>> resolveMap,
            final String inetHost, final Promise<U> promise, boolean resolveAll) {

        final Promise<U> earlyPromise = resolveMap.putIfAbsent(inetHost, promise);
        if (earlyPromise != null) {
            // Name resolution for the specified inetHost is in progress already.
            if (earlyPromise.isDone()) {
                transferResult(earlyPromise, promise);
            } else {
                earlyPromise.addListener(new FutureListener<U>() {
                    @Override
                    public void operationComplete(Future<U> f) throws Exception {
                        transferResult(f, promise);
                    }
                });
            }
        } else {
            try {
                if (resolveAll) {
                    @SuppressWarnings("unchecked")
                    final Promise<List<T>> castPromise = (Promise<List<T>>) promise;
                    delegate.resolveAll(inetHost, castPromise);
                } else {
                    @SuppressWarnings("unchecked")
                    final Promise<T> castPromise = (Promise<T>) promise;
                    delegate.resolve(inetHost, castPromise);
                }
            } finally {
                if (promise.isDone()) {
                    resolveMap.remove(inetHost);
                } else {
                    promise.addListener((FutureListener<U>) f -> resolveMap.remove(inetHost));
                }
            }
        }

        return promise;
    }

    private static <T> void transferResult(Future<T> src, Promise<T> dst) {
        if (src.isSuccess()) {
            dst.trySuccess(src.getNow());
        } else {
            dst.tryFailure(src.cause());
        }
    }

    @Override
    public String toString() {
        return StringUtil.simpleClassName(this) + '(' + delegate + ')';
    }
}
