package org.xbib.helianthus.client.http;

import static org.xbib.helianthus.common.util.Functions.voidFunction;

import io.netty.channel.EventLoop;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.Promise;
import org.xbib.helianthus.client.ClientOptionValue;
import org.xbib.helianthus.common.http.AggregatedHttpMessage;
import org.xbib.helianthus.common.http.HttpData;
import org.xbib.helianthus.common.http.HttpMethod;
import org.xbib.helianthus.common.http.HttpResponse;
import org.xbib.helianthus.common.util.CompletionActions;
import org.xbib.helianthus.internal.http.HelianthusHttpUtil;

import java.util.Arrays;

@SuppressWarnings("deprecation")
final class DefaultSimpleHttpClient implements SimpleHttpClient {

    private final DefaultHttpClient client;

    DefaultSimpleHttpClient(DefaultHttpClient client) {
        this.client = client;
    }

    @Override
    public Future<SimpleHttpResponse> execute(SimpleHttpRequest sReq) {
        final EventLoop eventLoop = client.eventLoop0();
        final Promise<SimpleHttpResponse> promise = eventLoop.newPromise();
        try {
            final AggregatedHttpMessage aReq = AggregatedHttpMessage.of(
                    HttpMethod.valueOf(sReq.method().name()),
                    sReq.uri().getPath(),
                    HttpData.of(sReq.content()));

            // Convert the headers.
            HelianthusHttpUtil.toHelianthus(sReq.headers(), aReq.headers());

            final HttpResponse res = client.execute(eventLoop, aReq);
            res.aggregate().handle(voidFunction((aRes, cause) -> {
                if (cause != null) {
                    promise.setFailure(cause);
                } else {
                    try {
                        final HttpData aContent = aRes.content();
                        final byte[] content;
                        if (aContent.offset() == 0 && aContent.length() == aContent.array().length) {
                            content = aContent.array();
                        } else {
                            content = Arrays.copyOfRange(aContent.array(), aContent.offset(),
                                    aContent.length());
                        }

                        final SimpleHttpResponse sRes = new SimpleHttpResponse(
                                HttpResponseStatus.valueOf(aRes.status().code()),
                                HelianthusHttpUtil.toNettyHttp1(aRes.headers()),
                                content);

                        promise.setSuccess(sRes);
                    } catch (Throwable t) {
                        promise.setFailure(t);
                    }
                }
            })).exceptionally(CompletionActions::log);
        } catch (Throwable t) {
            promise.setFailure(t);
        }

        return promise;
    }

    @Override
    public SimpleHttpClient withOptions(Iterable<ClientOptionValue<?>> additionalOptions) {
        return new DefaultSimpleHttpClient((DefaultHttpClient) client.withOptions(additionalOptions));
    }
}
