package org.xbib.helianthus.client.http;

import org.xbib.helianthus.client.ClientOptionDerivable;
import org.xbib.helianthus.common.http.AggregatedHttpMessage;
import org.xbib.helianthus.common.http.DefaultHttpRequest;
import org.xbib.helianthus.common.http.HttpData;
import org.xbib.helianthus.common.http.HttpHeaderNames;
import org.xbib.helianthus.common.http.HttpHeaders;
import org.xbib.helianthus.common.http.HttpMethod;
import org.xbib.helianthus.common.http.HttpRequest;
import org.xbib.helianthus.common.http.HttpResponse;

import java.nio.charset.Charset;

public interface HttpClient extends ClientOptionDerivable<HttpClient> {

    HttpResponse execute(HttpRequest req);

    default HttpResponse execute(AggregatedHttpMessage aggregatedReq) {
        final HttpHeaders headers = aggregatedReq.headers();
        final DefaultHttpRequest req = new DefaultHttpRequest(headers);
        final HttpData content = aggregatedReq.content();

        // Add content if not empty.
        if (!content.isEmpty()) {
            headers.setInt(HttpHeaderNames.CONTENT_LENGTH, content.length());
            req.write(content);
        }

        // Add trailing headers if not empty.
        final HttpHeaders trailingHeaders = aggregatedReq.trailingHeaders();
        if (!trailingHeaders.isEmpty()) {
            req.write(trailingHeaders);
        }

        req.close();
        return execute(req);
    }

    default HttpResponse execute(HttpHeaders headers) {
        return execute(AggregatedHttpMessage.of(headers));
    }

    default HttpResponse execute(HttpHeaders headers, HttpData content) {
        return execute(AggregatedHttpMessage.of(headers, content));
    }

    default HttpResponse execute(HttpHeaders headers, byte[] content) {
        return execute(AggregatedHttpMessage.of(headers, HttpData.of(content)));
    }

    default HttpResponse execute(HttpHeaders headers, String content) {
        return execute(AggregatedHttpMessage.of(headers, HttpData.ofUtf8(content)));
    }

    default HttpResponse execute(HttpHeaders headers, String content, Charset charset) {
        return execute(AggregatedHttpMessage.of(headers, HttpData.of(charset, content)));
    }

    default HttpResponse options(String path) {
        return execute(HttpHeaders.of(HttpMethod.OPTIONS, path));
    }

    default HttpResponse get(String path) {
        return execute(HttpHeaders.of(HttpMethod.GET, path));
    }

    default HttpResponse head(String path) {
        return execute(HttpHeaders.of(HttpMethod.HEAD, path));
    }

    default HttpResponse post(String path, HttpData content) {
        return execute(HttpHeaders.of(HttpMethod.POST, path), content);
    }

    default HttpResponse post(String path, byte[] content) {
        return execute(HttpHeaders.of(HttpMethod.POST, path), content);
    }

    default HttpResponse post(String path, String content) {
        return execute(HttpHeaders.of(HttpMethod.POST, path), HttpData.ofUtf8(content));
    }

    default HttpResponse post(String path, String content, Charset charset) {
        return execute(HttpHeaders.of(HttpMethod.POST, path), content, charset);
    }

    default HttpResponse put(String path, HttpData content) {
        return execute(HttpHeaders.of(HttpMethod.PUT, path), content);
    }

    default HttpResponse put(String path, byte[] content) {
        return execute(HttpHeaders.of(HttpMethod.PUT, path), content);
    }

    default HttpResponse put(String path, String content) {
        return execute(HttpHeaders.of(HttpMethod.PUT, path), HttpData.ofUtf8(content));
    }

    default HttpResponse put(String path, String content, Charset charset) {
        return execute(HttpHeaders.of(HttpMethod.PUT, path), content, charset);
    }

    default HttpResponse patch(String path, HttpData content) {
        return execute(HttpHeaders.of(HttpMethod.PATCH, path), content);
    }

    default HttpResponse patch(String path, byte[] content) {
        return execute(HttpHeaders.of(HttpMethod.PATCH, path), content);
    }

    default HttpResponse patch(String path, String content) {
        return execute(HttpHeaders.of(HttpMethod.PATCH, path), HttpData.ofUtf8(content));
    }

    default HttpResponse patch(String path, String content, Charset charset) {
        return execute(HttpHeaders.of(HttpMethod.PATCH, path), content, charset);
    }

    default HttpResponse delete(String path) {
        return execute(HttpHeaders.of(HttpMethod.DELETE, path));
    }

    default HttpResponse trace(String path) {
        return execute(HttpHeaders.of(HttpMethod.TRACE, path));
    }
}
