package org.xbib.helianthus.client.http;

import org.xbib.helianthus.client.Client;
import org.xbib.helianthus.client.ClientOptions;
import org.xbib.helianthus.client.Endpoint;
import org.xbib.helianthus.client.NonDecoratingClientFactory;
import org.xbib.helianthus.client.SessionOptions;
import org.xbib.helianthus.common.Scheme;
import org.xbib.helianthus.common.SerializationFormat;
import org.xbib.helianthus.common.SessionProtocol;
import org.xbib.helianthus.common.http.HttpRequest;
import org.xbib.helianthus.common.http.HttpResponse;

import java.net.URI;
import java.util.HashSet;
import java.util.Set;

public class HttpClientFactory extends NonDecoratingClientFactory {

    private static final Set<Scheme> SUPPORTED_SCHEMES;

    static {
        final Set<Scheme> set = new HashSet<>();
        for (SessionProtocol p : SessionProtocol.ofHttp()) {
            set.add(Scheme.of(SerializationFormat.NONE, p));
        }
        SUPPORTED_SCHEMES = set;
    }

    private final HttpClientDelegate delegate;

    public HttpClientFactory() {
        this(SessionOptions.DEFAULT);
    }

    public HttpClientFactory(SessionOptions options) {
        super(options);
        delegate = new HttpClientDelegate(baseBootstrap(), options);
    }

    @SuppressWarnings("deprecation")
    private static void validateClientType(Class<?> clientType) {
        if (clientType != HttpClient.class && clientType != SimpleHttpClient.class &&
                clientType != Client.class) {
            throw new IllegalArgumentException(
                    "clientType: " + clientType +
                            " (expected: " + HttpClient.class.getSimpleName() + ", " +
                            SimpleHttpClient.class.getSimpleName() + " or " +
                            Client.class.getSimpleName() + ')');
        }
    }

    @Override
    public Set<Scheme> supportedSchemes() {
        return SUPPORTED_SCHEMES;
    }

    @Override
    public <T> T newClient(URI uri, Class<T> clientType, ClientOptions options) {
        final Scheme scheme = validate(uri, clientType, options);

        validateClientType(clientType);

        final Client<HttpRequest, HttpResponse> delegate = options.decoration().decorate(
                HttpRequest.class, HttpResponse.class, new HttpClientDelegate(baseBootstrap(), options()));

        if (clientType == Client.class) {
            @SuppressWarnings("unchecked")
            final T castClient = (T) delegate;
            return castClient;
        }

        final Endpoint endpoint = newEndpoint(uri);

        if (clientType == HttpClient.class) {
            final HttpClient client = new DefaultHttpClient(
                    delegate, eventLoopSupplier(), scheme.sessionProtocol(), options, endpoint);


            @SuppressWarnings("unchecked")
            T castClient = (T) client;
            return castClient;
        } else {
            @SuppressWarnings("deprecation")
            final SimpleHttpClient client = new DefaultSimpleHttpClient(new DefaultHttpClient(
                    delegate, eventLoopSupplier(), scheme.sessionProtocol(), options, endpoint));

            @SuppressWarnings("unchecked")
            T castClient = (T) client;
            return castClient;
        }
    }

    @Override
    public void close() {
        delegate.close();
        super.close();
    }
}
