package org.xbib.helianthus.client;

import io.netty.channel.EventLoop;
import io.netty.util.Attribute;
import org.xbib.helianthus.common.Request;
import org.xbib.helianthus.common.RequestContext;
import org.xbib.helianthus.common.Response;
import org.xbib.helianthus.common.SessionProtocol;
import org.xbib.helianthus.common.http.DefaultHttpHeaders;
import org.xbib.helianthus.common.http.HttpHeaders;
import org.xbib.helianthus.common.util.SafeCloseable;

import java.net.URI;
import java.util.function.Function;

/**
 * A base class for implementing a user's entry point for sending a {@link Request}.
 *
 * <p>It provides the utility methods for easily forwarding a {@link Request} from a user to a {@link Client}.
 *
 * <p>Note that this class is not a subtype of {@link Client}, although its name may mislead.
 *
 * @param <I> the request type
 * @param <O> the response type
 */
public abstract class UserClient<I extends Request, O extends Response> implements ClientBuilderParams {

    private static final ThreadLocal<Function<HttpHeaders, HttpHeaders>> THREAD_LOCAL_HEADER_MANIPULATOR =
            new ThreadLocal<>();

    private final ClientBuilderParams params;
    private final Client<I, O> delegate;
    private final SessionProtocol sessionProtocol;
    private final Endpoint endpoint;

    protected UserClient(ClientBuilderParams params, Client<I, O> delegate,
                         SessionProtocol sessionProtocol, Endpoint endpoint) {
        this.params = params;

        this.delegate = delegate;
        this.sessionProtocol = sessionProtocol;
        this.endpoint = endpoint;
    }

    @Override
    public ClientFactory factory() {
        return params.factory();
    }

    @Override
    public URI uri() {
        return params.uri();
    }

    @Override
    public Class<?> clientType() {
        return params.clientType();
    }

    @Override
    public final ClientOptions options() {
        return params.options();
    }

    protected final EventLoop eventLoop() {
        return params.factory().eventLoopSupplier().get();
    }

    /**
     * Returns the {@link Client} that will process {@link Request}s.
     */
    @SuppressWarnings("unchecked")
    protected final <U extends Client<I, O>> U delegate() {
        return (U) delegate;
    }

    protected final SessionProtocol sessionProtocol() {
        return sessionProtocol;
    }

    protected final Endpoint endpoint() {
        return endpoint;
    }

    /**
     * Executes the specified {@link Request} via {@link #delegate()}.
     *
     * @param method the method of the {@link Request}
     * @param path the path of the {@link Request} URI
     * @param fragment the fragment part of the {@link Request} URI
     * @param req the {@link Request}
     * @param fallback the fallback response {@link Function} to use when
     *                 {@link Client#execute(ClientRequestContext, Request)} of {@link #delegate()} throws
     *                 an exception instead of returning an error response
     */
    protected final O execute(String method, String path, String fragment, I req, Function<Throwable, O> fallback) {
        return execute(eventLoop(), method, path, fragment, req, fallback);
    }

    @SuppressWarnings("try")
    protected final O execute(EventLoop eventLoop, String method, String path, String fragment,
                              I req, Function<Throwable, O> fallback) {
        final ClientRequestContext ctx = new DefaultClientRequestContext(eventLoop,
                sessionProtocol, endpoint, method, path, fragment, options(), req);
        try (SafeCloseable ignored = RequestContext.push(ctx)) {
            runThreadLocalHeaderManipulator(ctx);
            return delegate().execute(ctx, req);
        } catch (Throwable cause) {
            ctx.logBuilder().endResponse(cause);
            return fallback.apply(cause);
        }
    }

    private static void runThreadLocalHeaderManipulator(ClientRequestContext ctx) {
        final Function<HttpHeaders, HttpHeaders> manipulator = THREAD_LOCAL_HEADER_MANIPULATOR.get();
        if (manipulator == null) {
            return;
        }
        final Attribute<HttpHeaders> attr = ctx.attr(ClientRequestContext.HTTP_HEADERS);
        final HttpHeaders headers = attr.get();
        attr.set(manipulator.apply(headers != null ? headers : new DefaultHttpHeaders()));
    }
}
