package org.xbib.helianthus.client.http;

import io.netty.bootstrap.Bootstrap;
import org.xbib.helianthus.client.Client;
import org.xbib.helianthus.client.ClientFactory;
import org.xbib.helianthus.client.ClientOptions;
import org.xbib.helianthus.client.DefaultClientBuilderParams;
import org.xbib.helianthus.client.Endpoint;
import org.xbib.helianthus.client.NonDecoratingClientFactory;
import org.xbib.helianthus.client.SessionOptions;
import org.xbib.helianthus.common.Scheme;
import org.xbib.helianthus.common.SerializationFormat;
import org.xbib.helianthus.common.SessionProtocol;
import org.xbib.helianthus.common.http.HttpRequest;
import org.xbib.helianthus.common.http.HttpResponse;

import java.net.URI;
import java.util.HashSet;
import java.util.Set;

/**
 * A {@link ClientFactory} that creates an HTTP client.
 */
public class HttpClientFactory extends NonDecoratingClientFactory {

    private static final Set<Scheme> SUPPORTED_SCHEMES;

    static {
        final Set<Scheme> set = new HashSet<>();
        for (SessionProtocol p : SessionProtocol.ofHttp()) {
            set.add(Scheme.of(SerializationFormat.NONE, p));
        }
        SUPPORTED_SCHEMES = set;
    }

    private final HttpClientDelegate delegate;

    public HttpClientFactory() {
        this(SessionOptions.DEFAULT, false);
    }

    public HttpClientFactory(SessionOptions options) {
        this(options, false);
    }

    public HttpClientFactory(SessionOptions options, boolean useDaemonThreads) {
        super(options, useDaemonThreads);
        delegate = new HttpClientDelegate(this);
    }

    @Override
    public Set<Scheme> supportedSchemes() {
        return SUPPORTED_SCHEMES;
    }

    @Override
    public Bootstrap newBootstrap() {
        return super.newBootstrap();
    }

    @Override
    public <T> T newClient(URI uri, Class<T> clientType, ClientOptions options) {
        final Scheme scheme = validate(uri, clientType, options);
        validateClientType(clientType);
        final Client<HttpRequest, HttpResponse> delegate = options.decoration().decorate(HttpRequest.class,
                HttpResponse.class, new HttpClientDelegate(this));
        if (clientType == Client.class) {
            @SuppressWarnings("unchecked")
            final T castClient = (T) delegate;
            return castClient;
        }
        final Endpoint endpoint = newEndpoint(uri);
        final HttpClient client = newHttpClient(uri, scheme, endpoint, options, delegate);

        @SuppressWarnings("unchecked")
        T castClient = (T) client;
        return castClient;
    }

    private DefaultHttpClient newHttpClient(URI uri, Scheme scheme, Endpoint endpoint, ClientOptions options,
                                            Client<HttpRequest, HttpResponse> delegate) {
        return new DefaultHttpClient(new DefaultClientBuilderParams(this, uri, HttpClient.class, options),
                delegate, scheme.sessionProtocol(), endpoint);
    }

    @SuppressWarnings("deprecation")
    private static void validateClientType(Class<?> clientType) {
        if (clientType != HttpClient.class && clientType != Client.class) {
            throw new IllegalArgumentException("clientType: " + clientType +
                            " (expected: " + HttpClient.class.getSimpleName() + ", " +
                            Client.class.getSimpleName() + ')');
        }
    }

    @Override
    public void close() {
        delegate.close();
        super.close();
    }
}
