package org.xbib.helianthus.client.http;

import static java.util.Objects.requireNonNull;

import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.EventLoop;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.Promise;
import org.xbib.helianthus.client.SessionOptions;
import org.xbib.helianthus.client.SessionProtocolNegotiationCache;
import org.xbib.helianthus.client.SessionProtocolNegotiationException;
import org.xbib.helianthus.client.pool.PoolKey;
import org.xbib.helianthus.common.SessionProtocol;

import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.Collections;
import java.util.EnumMap;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

class HttpSessionChannelFactory implements Function<PoolKey, Future<Channel>> {

    private final Bootstrap baseBootstrap;

    private final EventLoop eventLoop;

    private final Map<SessionProtocol, Bootstrap> bootstrapMap;

    private final SessionOptions options;

    HttpSessionChannelFactory(Bootstrap bootstrap, SessionOptions options) {
        baseBootstrap = requireNonNull(bootstrap);
        eventLoop = (EventLoop) bootstrap.config().group();
        bootstrapMap = Collections.synchronizedMap(new EnumMap<>(SessionProtocol.class));
        this.options = options;
    }

    @Override
    public Future<Channel> apply(PoolKey key) {
        final InetSocketAddress remoteAddress = key.remoteAddress();
        final SessionProtocol protocol = key.sessionProtocol();
        if (SessionProtocolNegotiationCache.isUnsupported(remoteAddress, protocol)) {
            // Fail immediately if it is sure that the remote address doesn't support the requested protocol.
            return eventLoop.newFailedFuture(new SessionProtocolNegotiationException(protocol,
                    "previously failed negotiation"));
        }
        final Promise<Channel> sessionPromise = eventLoop.newPromise();
        connect(remoteAddress, protocol, sessionPromise);
        return sessionPromise;
    }

    void connect(SocketAddress remoteAddress, SessionProtocol protocol, Promise<Channel> sessionPromise) {
        final Bootstrap bootstrap = bootstrap(protocol);
        final ChannelFuture connectFuture = bootstrap.connect(remoteAddress);
        connectFuture.addListener((ChannelFuture future) -> {
            if (future.isSuccess()) {
                initSession(protocol, future, sessionPromise);
            } else {
                sessionPromise.setFailure(future.cause());
            }
        });
    }

    private Bootstrap bootstrap(SessionProtocol sessionProtocol) {
        return bootstrapMap.computeIfAbsent(sessionProtocol, sp -> {
            Bootstrap bs = baseBootstrap.clone();
            bs.handler(new ChannelInitializer<Channel>() {
                @Override
                protected void initChannel(Channel ch) throws Exception {
                    ch.pipeline().addLast(new HttpClientPipelineConfigurator(sp, options));
                }
            });
            return bs;
        });
    }

    private void initSession(SessionProtocol protocol, ChannelFuture connectFuture,
                             Promise<Channel> sessionPromise) {
        if (!connectFuture.isSuccess()) {
            throw new IllegalStateException();
        }
        final Channel ch = connectFuture.channel();
        final EventLoop eventLoop = ch.eventLoop();
        if (!eventLoop.inEventLoop()) {
            throw new IllegalStateException();
        }
        final ScheduledFuture<?> timeoutFuture = eventLoop.schedule(() -> {
            if (sessionPromise.tryFailure(new SessionProtocolNegotiationException(
                    protocol, "connection established, but session creation timed out: " + ch))) {
                ch.close();
            }
        }, options.connectTimeoutMillis(), TimeUnit.MILLISECONDS);
        ch.pipeline().addLast(new HttpSessionHandler(this, ch, sessionPromise, timeoutFuture));
    }
}
