package org.xbib.helianthus.client;

import io.netty.channel.EventLoop;
import io.netty.channel.EventLoopGroup;
import org.xbib.helianthus.common.Scheme;

import java.net.URI;
import java.text.MessageFormat;
import java.util.Set;
import java.util.function.Supplier;
import java.util.logging.Logger;

/**
 * Creates and manages clients.
 *
 * <h3>Life cycle of the default {@link ClientFactory}</h3>
 * <p>
 * {@link Clients} or {@link ClientBuilder} uses {@link #DEFAULT}, the default {@link ClientFactory},
 * unless you specified a {@link ClientFactory} explicitly. Calling {@link #close()} on the default
 * {@link ClientFactory} won't terminate its I/O threads and release other related resources unlike
 * other {@link ClientFactory} to protect itself from accidental premature termination.
 * </p><p>
 * Instead, when the current {@link ClassLoader} is {@linkplain ClassLoader#getSystemClassLoader() the system
 * class loader}, a {@link Runtime#addShutdownHook(Thread) shutdown hook} is registered so that they are
 * released when the JVM exits.
 * </p><p>
 * If you are in an environment managed by a container or you desire the early termination of the default
 * {@link ClientFactory}, use {@link #closeDefault()}.
 * </p>
 */
public interface ClientFactory extends AutoCloseable {

    /**
     * The default {@link ClientFactory} implementation.
     */
    ClientFactory DEFAULT = new AllInOneClientFactory();

    /**
     * Closes the default {@link ClientFactory}.
     */
    static void closeDefault() {
        Logger.getLogger(ClientFactory.class.getName()).fine(
                MessageFormat.format("default close {0}", ClientFactory.class.getName()));
        ((AllInOneClientFactory) DEFAULT).doClose();
    }

    Set<Scheme> supportedSchemes();

    SessionOptions options();

    /**
     * Returns the {@link EventLoopGroup} being used by this {@link ClientFactory}. Can be used to, e.g.,
     * schedule a periodic task without creating a separate event loop.
     */
    EventLoopGroup eventLoopGroup();

    Supplier<EventLoop> eventLoopSupplier();

    /**
     * Creates a new client that connects to the specified {@code uri}.
     *
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptionValue}s
     */
    <T> T newClient(String uri, Class<T> clientType, ClientOptionValue<?>... options);

    /**
     * Creates a new client that connects to the specified {@code uri}.
     *
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptions}
     */
    <T> T newClient(String uri, Class<T> clientType, ClientOptions options);

    /**
     * Creates a new client that connects to the specified {@link URI} using the default
     * {@link ClientFactory}.
     *
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptionValue}s
     */
    <T> T newClient(URI uri, Class<T> clientType, ClientOptionValue<?>... options);

    /**
     * Creates a new client that connects to the specified {@link URI} using the default
     * {@link ClientFactory}.
     *
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptions}
     */
    <T> T newClient(URI uri, Class<T> clientType, ClientOptions options);

    /**
     * Closes all clients managed by this factory and shuts down the {@link EventLoopGroup}
     * created implicitly by this factory.
     */
    @Override
    void close();
}
