package org.xbib.helianthus.client;

import static java.util.Objects.requireNonNull;
import static org.xbib.helianthus.client.ClientOption.DECORATION;
import static org.xbib.helianthus.client.ClientOption.DEFAULT_MAX_RESPONSE_LENGTH;
import static org.xbib.helianthus.client.ClientOption.DEFAULT_RESPONSE_TIMEOUT_MILLIS;
import static org.xbib.helianthus.client.ClientOption.DEFAULT_WRITE_TIMEOUT_MILLIS;
import static org.xbib.helianthus.client.ClientOption.HTTP_HEADERS;

import io.netty.handler.codec.http2.HttpConversionUtil.ExtensionHeaderNames;
import io.netty.util.AsciiString;
import org.xbib.helianthus.common.http.DefaultHttpHeaders;
import org.xbib.helianthus.common.http.HttpHeaderNames;
import org.xbib.helianthus.common.http.HttpHeaders;
import org.xbib.helianthus.common.util.AbstractOptions;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

/**
 * A set of {@link ClientOption}s and their respective values.
 */
public final class ClientOptions extends AbstractOptions {

    private static final Long DEFAULT_DEFAULT_WRITE_TIMEOUT_MILLIS = Duration.ofSeconds(1).toMillis();

    private static final Long DEFAULT_DEFAULT_RESPONSE_TIMEOUT_MILLIS = Duration.ofSeconds(10).toMillis();

    private static final Long DEFAULT_DEFAULT_MAX_RESPONSE_LENGTH = 10L * 1024 * 1024; // 10 MB

    @SuppressWarnings("deprecation")
    private static final Collection<AsciiString> BLACKLISTED_HEADER_NAMES =
            Collections.unmodifiableCollection(Arrays.asList(
                    HttpHeaderNames.AUTHORITY,
                    HttpHeaderNames.CONNECTION,
                    HttpHeaderNames.HOST,
                    HttpHeaderNames.KEEP_ALIVE,
                    HttpHeaderNames.METHOD,
                    HttpHeaderNames.PATH,
                    HttpHeaderNames.PROXY_CONNECTION,
                    HttpHeaderNames.SCHEME,
                    HttpHeaderNames.STATUS,
                    HttpHeaderNames.TRANSFER_ENCODING,
                    HttpHeaderNames.UPGRADE,
                    HttpHeaderNames.USER_AGENT,
                    ExtensionHeaderNames.PATH.text(),
                    ExtensionHeaderNames.SCHEME.text(),
                    ExtensionHeaderNames.STREAM_DEPENDENCY_ID.text(),
                    ExtensionHeaderNames.STREAM_ID.text(),
                    ExtensionHeaderNames.STREAM_PROMISE_ID.text()));

    private static final ClientOptionValue<?>[] DEFAULT_OPTIONS = {
            DEFAULT_WRITE_TIMEOUT_MILLIS.newValue(DEFAULT_DEFAULT_WRITE_TIMEOUT_MILLIS),
            DEFAULT_RESPONSE_TIMEOUT_MILLIS.newValue(DEFAULT_DEFAULT_RESPONSE_TIMEOUT_MILLIS),
            DEFAULT_MAX_RESPONSE_LENGTH.newValue(DEFAULT_DEFAULT_MAX_RESPONSE_LENGTH),
            DECORATION.newValue(ClientDecoration.NONE),
            HTTP_HEADERS.newValue(HttpHeaders.EMPTY_HEADERS),

    };

    /**
     * The default {@link ClientOptions}.
     */
    public static final ClientOptions DEFAULT = new ClientOptions(DEFAULT_OPTIONS);

    private ClientOptions(ClientOptionValue<?>... options) {
        super(ClientOptions::filterValue, options);
    }

    private ClientOptions(ClientOptions clientOptions, ClientOptionValue<?>... options) {
        super(ClientOptions::filterValue, clientOptions, options);
    }

    private ClientOptions(ClientOptions clientOptions, Iterable<ClientOptionValue<?>> options) {
        super(ClientOptions::filterValue, clientOptions, options);
    }

    private ClientOptions(ClientOptions clientOptions, ClientOptions options) {
        super(clientOptions, options);
    }

    /**
     * Returns the {@link ClientOptions} with the specified {@link ClientOptionValue}s.
     */
    public static ClientOptions of(ClientOptionValue<?>... options) {
        requireNonNull(options, "options");
        if (options.length == 0) {
            return DEFAULT;
        }
        return new ClientOptions(DEFAULT, options);
    }

    /**
     * Returns the {@link ClientOptions} with the specified {@link ClientOptionValue}s.
     */
    public static ClientOptions of(Iterable<ClientOptionValue<?>> options) {
        return new ClientOptions(DEFAULT, options);
    }

    /**
     * Merges the specified {@link ClientOptions} and {@link ClientOptionValue}s.
     *
     * @return the merged {@link ClientOptions}
     */
    public static ClientOptions of(ClientOptions baseOptions, ClientOptionValue<?>... options) {
        // TODO(trustin): Reduce the cost of creating a derived ClientOptions.
        requireNonNull(baseOptions, "baseOptions");
        requireNonNull(options, "options");
        if (options.length == 0) {
            return baseOptions;
        }
        return new ClientOptions(baseOptions, options);
    }

    /**
     * Merges the specified {@link ClientOptions} and {@link ClientOptionValue}s.
     *
     * @return the merged {@link ClientOptions}
     */
    public static ClientOptions of(ClientOptions baseOptions, Iterable<ClientOptionValue<?>> options) {
        // TODO(trustin): Reduce the cost of creating a derived ClientOptions.
        requireNonNull(baseOptions, "baseOptions");
        requireNonNull(options, "options");
        return new ClientOptions(baseOptions, options);
    }

    public static ClientOptions of(ClientOptions baseOptions, ClientOptions options) {
        // TODO(trustin): Reduce the cost of creating a derived ClientOptions.
        requireNonNull(baseOptions, "baseOptions");
        requireNonNull(options, "options");
        return new ClientOptions(baseOptions, options);
    }

    private static <T> ClientOptionValue<T> filterValue(ClientOptionValue<T> optionValue) {
        requireNonNull(optionValue, "optionValue");

        ClientOption<?> option = optionValue.option();
        T value = optionValue.value();

        if (option == HTTP_HEADERS) {
            @SuppressWarnings("unchecked")
            ClientOption<HttpHeaders> castOption = (ClientOption<HttpHeaders>) option;
            @SuppressWarnings("unchecked")
            ClientOptionValue<T> castOptionValue =
                    (ClientOptionValue<T>) castOption.newValue(filterHttpHeaders((HttpHeaders) value));
            optionValue = castOptionValue;
        }
        return optionValue;
    }

    private static HttpHeaders filterHttpHeaders(HttpHeaders headers) {
        requireNonNull(headers, "headers");
        for (AsciiString name : BLACKLISTED_HEADER_NAMES) {
            if (headers.contains(name)) {
                throw new IllegalArgumentException("unallowed header name: " + name);
            }
        }

        return new DefaultHttpHeaders().add(headers).asImmutable();
    }

    /**
     * Returns the value of the specified {@link ClientOption}.
     *
     * @return the value of the {@link ClientOption}, or
     * {@link Optional#empty()} if the default value of the specified {@link ClientOption} is not
     * available
     */
    public <T> Optional<T> get(ClientOption<T> option) {
        return get0(option);
    }

    /**
     * Returns the value of the specified {@link ClientOption}.
     *
     * @return the value of the {@link ClientOption}, or
     * {@code defaultValue} if the specified {@link ClientOption} is not set.
     */
    public <T> T getOrElse(ClientOption<T> option, T defaultValue) {
        return getOrElse0(option, defaultValue);
    }

    /**
     * Converts this {@link ClientOptions} to a {@link Map}.
     */
    public Map<ClientOption<Object>, ClientOptionValue<Object>> asMap() {
        return asMap0();
    }

    /**
     * Returns the default timeout of a server reply to a client call.
     */
    public long defaultResponseTimeoutMillis() {
        return getOrElse(DEFAULT_RESPONSE_TIMEOUT_MILLIS, DEFAULT_DEFAULT_RESPONSE_TIMEOUT_MILLIS);
    }

    /**
     * Returns the default timeout of a socket write.
     */
    public long defaultWriteTimeoutMillis() {
        return getOrElse(DEFAULT_WRITE_TIMEOUT_MILLIS, DEFAULT_DEFAULT_WRITE_TIMEOUT_MILLIS);
    }

    /**
     * Returns the maximum allowed length of a server response.
     */
    @SuppressWarnings("unchecked")
    public long defaultMaxResponseLength() {
        return getOrElse(DEFAULT_MAX_RESPONSE_LENGTH, DEFAULT_DEFAULT_MAX_RESPONSE_LENGTH);
    }

    /**
     * Returns the {@link Function}s that decorate the components of a client.
     */
    public ClientDecoration decoration() {
        return getOrElse(DECORATION, ClientDecoration.NONE);
    }
}

