package org.xbib.helianthus.client.http;

import static java.util.Objects.requireNonNull;
import static org.xbib.helianthus.common.SessionProtocol.H2;
import static org.xbib.helianthus.common.SessionProtocol.H2C;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.channel.Channel;
import io.netty.channel.ChannelDuplexHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http2.Http2ConnectionHandler;
import io.netty.handler.codec.http2.Http2Settings;
import io.netty.handler.ssl.SslCloseCompletionEvent;
import io.netty.handler.ssl.SslHandshakeCompletionEvent;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.concurrent.Promise;
import org.xbib.helianthus.client.ClientRequestContext;
import org.xbib.helianthus.client.SessionProtocolNegotiationException;
import org.xbib.helianthus.common.ClosedSessionException;
import org.xbib.helianthus.common.SessionProtocol;
import org.xbib.helianthus.common.http.HttpRequest;
import org.xbib.helianthus.common.util.Exceptions;
import org.xbib.helianthus.internal.InboundTrafficController;
import org.xbib.helianthus.internal.http.Http1ObjectEncoder;
import org.xbib.helianthus.internal.http.Http2ObjectEncoder;
import org.xbib.helianthus.internal.http.HttpObjectEncoder;

import java.text.MessageFormat;
import java.util.concurrent.ScheduledFuture;
import java.util.logging.Level;
import java.util.logging.Logger;

final class HttpSessionHandler extends ChannelDuplexHandler implements HttpSession {

    private static final Logger logger = Logger.getLogger(HttpSessionHandler.class.getName());

    /**
     * 2^29 - We could have used 2^30 but this should be large enough.
     */
    private static final int MAX_NUM_REQUESTS_SENT = 536870912;

    private final HttpSessionChannelFactory channelFactory;

    private final Channel channel;

    private final Promise<Channel> sessionPromise;

    private final ScheduledFuture<?> sessionTimeoutFuture;

    /**
     * Whether the current channel is active or not
     **/
    private volatile boolean active;

    /**
     * The current negotiated {@link SessionProtocol}
     */
    private SessionProtocol protocol;

    private HttpResponseDecoder responseDecoder;

    private HttpObjectEncoder requestEncoder;

    /**
     * The number of requests sent. Disconnects when it reaches at {@link #MAX_NUM_REQUESTS_SENT}.
     */
    private int numRequestsSent;

    /**
     * {@code true} if the protocol upgrade to HTTP/2 has failed.
     * If set to {@code true}, another connection attempt will follow.
     */
    private boolean needsRetryWithH1C;

    HttpSessionHandler(HttpSessionChannelFactory channelFactory, Channel channel,
                       Promise<Channel> sessionPromise, ScheduledFuture<?> sessionTimeoutFuture) {
        this.channelFactory = requireNonNull(channelFactory, "channelFactory");
        this.channel = requireNonNull(channel, "channel");
        this.sessionPromise = requireNonNull(sessionPromise, "sessionPromise");
        this.sessionTimeoutFuture = requireNonNull(sessionTimeoutFuture, "sessionTimeoutFuture");
    }

    @Override
    public SessionProtocol protocol() {
        return protocol;
    }

    @Override
    public InboundTrafficController inboundTrafficController() {
        return responseDecoder.inboundTrafficController();
    }

    @Override
    public boolean hasUnfinishedResponses() {
        return responseDecoder.hasUnfinishedResponses();
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public boolean invoke(ClientRequestContext ctx, HttpRequest req, DecodedHttpResponse res) {
        if (!res.isOpen()) {
            // The response has been closed even before its request is sent.
            req.abort();
            return true;
        }
        final long writeTimeoutMillis = ctx.writeTimeoutMillis();
        final long responseTimeoutMillis = ctx.responseTimeoutMillis();
        final long maxContentLength = ctx.maxResponseLength();
        final int numRequestsSent = ++this.numRequestsSent;
        final HttpResponseDecoder.HttpResponseWrapper wrappedRes =
                responseDecoder.addResponse(numRequestsSent, req, res, ctx.logBuilder(),
                        responseTimeoutMillis, maxContentLength);
        req.subscribe(new HttpRequestSubscriber(channel, requestEncoder,
                        numRequestsSent, req, wrappedRes, ctx.logBuilder(),
                        writeTimeoutMillis), channel.eventLoop());
        if (numRequestsSent >= MAX_NUM_REQUESTS_SENT) {
            responseDecoder.disconnectWhenFinished();
            return false;
        } else {
            return true;
        }
    }

    @Override
    public void retryWithH1C() {
        needsRetryWithH1C = true;
    }

    @Override
    public void deactivate() {
        active = false;
    }

    @Override
    public void handlerAdded(ChannelHandlerContext ctx) throws Exception {
        active = ctx.channel().isActive();
    }

    @Override
    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        active = true;
    }

    @Override
    public void channelRead(ChannelHandlerContext ctx, Object msg) throws Exception {
        if (!(msg instanceof Http2Settings)) {
            try {
                final String typeInfo;
                if (msg instanceof ByteBuf) {
                    typeInfo = msg + " HexDump: " + ByteBufUtil.hexDump((ByteBuf) msg);
                } else {
                    typeInfo = String.valueOf(msg);
                }
                throw new IllegalStateException("unexpected message type: " + msg.getClass() + " " + typeInfo);
            } finally {
                ReferenceCountUtil.release(msg);
            }
        }
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
        if (evt instanceof SessionProtocol) {
            if (protocol != null) {
                throw new IllegalStateException();
            }
            if (responseDecoder != null) {
                throw new IllegalStateException();
            }
            sessionTimeoutFuture.cancel(false);
            // Set the current protocol and its associated WaitsHolder implementation.
            final SessionProtocol protocol = (SessionProtocol) evt;
            this.protocol = protocol;
            switch (protocol) {
                case H1:
                case H1C:
                    requestEncoder = new Http1ObjectEncoder(false);
                    responseDecoder = ctx.pipeline().get(Http1ResponseDecoder.class);
                    break;
                case H2:
                case H2C:
                    final Http2ConnectionHandler handler = ctx.pipeline().get(Http2ConnectionHandler.class);
                    requestEncoder = new Http2ObjectEncoder(handler.encoder());
                    responseDecoder = ctx.pipeline().get(Http2ClientConnectionHandler.class).responseDecoder();
                    break;
                default:
                    throw new IllegalStateException(); // Should never reach here.
            }
            if (!sessionPromise.trySuccess(ctx.channel())) {
                // Session creation has been failed already; close the connection.
                ctx.close();
            }
            return;
        }
        if (evt instanceof SessionProtocolNegotiationException) {
            sessionTimeoutFuture.cancel(false);
            sessionPromise.tryFailure((SessionProtocolNegotiationException) evt);
            ctx.close();
            return;
        }
        if (evt instanceof SslHandshakeCompletionEvent) {
            SslHandshakeCompletionEvent event = (SslHandshakeCompletionEvent)evt;
            if (!event.isSuccess()) {
                logger.warning("SSL handshake completion event, but no success");
                sessionTimeoutFuture.cancel(false);
                sessionPromise.tryFailure(event.cause());
                ctx.close();
                return;
            }
        }
        if (evt instanceof SslCloseCompletionEvent) {
            SslCloseCompletionEvent event = (SslCloseCompletionEvent)evt;
            if (!event.isSuccess()) {
                sessionTimeoutFuture.cancel(false);
                ctx.close();
                if (protocol == H2 || protocol == H2C) {
                    retryWithH1C();
                }
                return;
            }
        }
        logger.log(Level.WARNING, MessageFormat.format("{0} Unexpected user event {1}", ctx.channel(), evt));
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        active = false;
        // Protocol upgrade has failed, but needs to retry.
        if (needsRetryWithH1C) {
            sessionTimeoutFuture.cancel(false);
            channelFactory.connect(ctx.channel().remoteAddress(), SessionProtocol.H1C, sessionPromise);
        } else {
            // Fail all pending responses.
            failUnfinishedResponses(ClosedSessionException.get());
            // Cancel the timeout and reject the sessionPromise just in case the connection has been closed
            // even before the session protocol negotiation is done.
            sessionTimeoutFuture.cancel(false);
            sessionPromise.tryFailure(ClosedSessionException.get());
        }
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
        Exceptions.logIfUnexpected(logger, ctx.channel(), protocol(), cause);
        if (ctx.channel().isActive()) {
            ctx.close();
        }
    }

    private void failUnfinishedResponses(Throwable e) {
        final HttpResponseDecoder responseDecoder = this.responseDecoder;
        if (responseDecoder == null) {
            return;
        }
        responseDecoder.failUnfinishedResponses(e);
    }
}
