package org.xbib.helianthus.common.http;

import org.reactivestreams.Publisher;
import org.xbib.helianthus.common.Request;
import org.xbib.helianthus.common.stream.StreamMessage;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

public interface HttpRequest extends Request, StreamMessage<HttpObject> {

    static HttpRequest of(HttpHeaders headers, Publisher<? extends HttpObject> publisher) {
        return new PublisherBasedHttpRequest(headers, true, publisher);
    }

    /**
     * Returns a new {@link HttpRequest} with empty content.
     */
    static HttpRequest of(HttpHeaders headers) {
        final DefaultHttpRequest req = new DefaultHttpRequest(headers);
        req.close();
        return req;
    }

    HttpHeaders headers();

    boolean isKeepAlive();

    default String scheme() {
        return headers().scheme();
    }

    default HttpRequest scheme(String scheme) {
        headers().scheme(scheme);
        return this;
    }

    default HttpMethod method() {
        return headers().method();
    }

    default HttpRequest method(HttpMethod method) {
        headers().method(method);
        return this;
    }

    default String path() {
        return headers().path();
    }

    default HttpRequest path(String path) {
        headers().path(path);
        return this;
    }

    default String authority() {
        return headers().authority();
    }

    default HttpRequest authority(String authority) {
        headers().authority(authority);
        return this;
    }

    /**
     * Aggregates the request. The returned {@link CompletableFuture} will be notified when the content and
     * the trailing headers of the request is received fully.
     */
    default CompletableFuture<AggregatedHttpMessage> aggregate() {
        final CompletableFuture<AggregatedHttpMessage> future = new CompletableFuture<>();
        subscribe(new HttpRequestAggregator(this, future));
        return future;
    }

    /**
     * Aggregates the request. The returned {@link CompletableFuture} will be notified when the content and
     * the trailing headers of the request is received fully.
     */
    default CompletableFuture<AggregatedHttpMessage> aggregate(Executor executor) {
        final CompletableFuture<AggregatedHttpMessage> future = new CompletableFuture<>();
        subscribe(new HttpRequestAggregator(this, future), executor);
        return future;
    }
}
