package org.xbib.helianthus.common;

import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;

/**
 * Session-level protocol that provides facilities such as framing and flow control.
 */
public enum SessionProtocol {
    /**
     * HTTP (cleartext, HTTP/2 preferred)
     */
    HTTP(false, "http", "HTTP/1.1", false, 80),
    /**
     * HTTP over TLS (over TLS, HTTP/2 preferred)
     */
    HTTPS(true, "https", "HTTP/1.1", false, 443),
    /**
     * HTTP/1 (over TLS)
     */
    H1(true, "h1", "HTTP/1.1", false, 443),
    /**
     * HTTP/1 (cleartext)
     */
    H1C(false, "h1c", "HTTP/1.1", false, 80),
    /**
     * HTTP/2 (over TLS)
     */
    H2(true, "h2", "HTTP/2", true, 443),
    /**
     * HTTP/2 (cleartext)
     */
    H2C(false, "h2c", "HTTP/2", true, 80);

    private static final Set<SessionProtocol> HTTP_PROTOCOLS = Collections.unmodifiableSet(
            EnumSet.of(HTTP, HTTPS, H1, H1C, H2, H2C));
    private final boolean useTls;
    private final String uriText;
    private final String protocolVersionText;
    private final boolean isMultiplex;
    private final int defaultPort;

    SessionProtocol(boolean useTls, String uriText, String protocolVersionText, boolean isMultiplex, int defaultPort) {
        this.useTls = useTls;
        this.uriText = uriText;
        this.protocolVersionText = protocolVersionText;
        this.isMultiplex = isMultiplex;
        this.defaultPort = defaultPort;
    }

    /**
     * Returns the set of all known HTTP session protocols. This method is useful when determining if a
     * {@link SessionProtocol} is HTTP or not.
     * e.g. {@code if (SessionProtocol.ofHttp().contains(proto)) { ... }}
     */
    public static Set<SessionProtocol> ofHttp() {
        return HTTP_PROTOCOLS;
    }

    /**
     * Returns {@code true} if and only if this protocol uses TLS as its transport-level security layer.
     */
    public boolean isTls() {
        return useTls;
    }

    /**
     * Returns the textual representation of this format for use in a {@link Scheme}.
     */
    public String uriText() {
        return uriText;
    }

    /**
     * Return a protocol/version text for logging.
     */
    public String protocolVersionText() {
        return protocolVersionText;
    }

    /**
     * Returns {@code true} if and only if this protocol can multiplex a single transport-layer connection into
     * more than one stream.
     */
    public boolean isMultiplex() {
        return isMultiplex;
    }

    /**
     * Returns the default INET port number of this protocol.
     */
    public int defaultPort() {
        return defaultPort;
    }
}
