package org.xbib.helianthus.common.http;

import static java.util.Objects.requireNonNull;

import org.xbib.helianthus.common.util.CollectionUtil;

import java.util.Collections;
import java.util.List;

public interface AggregatedHttpMessage {

    static AggregatedHttpMessage of(HttpMethod method, String path) {
        return of(HttpHeaders.of(method, path));
    }

    static AggregatedHttpMessage of(HttpMethod method, String path, HttpData content) {
        return of(HttpHeaders.of(method, path), content);
    }

    static AggregatedHttpMessage of(int statusCode) {
        return of(HttpStatus.valueOf(statusCode));
    }

    static AggregatedHttpMessage of(HttpStatus status) {
        return of(HttpHeaders.of(status));
    }

    static AggregatedHttpMessage of(HttpStatus status, HttpData content) {
        return of(HttpHeaders.of(status), content);
    }

    static AggregatedHttpMessage of(HttpHeaders headers, HttpStatus status, HttpData content) {
        return of(headers.status(status), content);
    }

    static AggregatedHttpMessage of(HttpHeaders headers) {
        return of(headers, HttpData.EMPTY_DATA, HttpHeaders.EMPTY_HEADERS, null);
    }

    static AggregatedHttpMessage of(HttpHeaders headers, HttpData content) {
        return of(headers, content, HttpHeaders.EMPTY_HEADERS, null);
    }

    static AggregatedHttpMessage of(HttpHeaders headers, HttpData content, HttpHeaders trailingHeaders, String followUrl) {
        return of(Collections.emptyList(), headers, content, trailingHeaders, followUrl);
    }

    static AggregatedHttpMessage of(Iterable<HttpHeaders> informationals, HttpHeaders headers,
                                    HttpData content, HttpHeaders trailingHeaders, String followUrl) {

        requireNonNull(informationals, "informationals");
        requireNonNull(headers, "headers");
        requireNonNull(content, "content");
        requireNonNull(trailingHeaders, "trailingHeaders");

        return new DefaultAggregatedHttpMessage(CollectionUtil.makeList(informationals),
                headers, content, trailingHeaders, followUrl);
    }

    List<HttpHeaders> informationals();

    HttpHeaders headers();

    HttpHeaders trailingHeaders();

    HttpData content();

    String followUrl();

    default String scheme() {
        return headers().scheme();
    }

    default HttpMethod method() {
        return headers().method();
    }

    default String path() {
        return headers().path();
    }

    default String authority() {
        return headers().authority();
    }

    default HttpStatus status() {
        return headers().status();
    }
}
