package org.xbib.helianthus.common.util;

import io.netty.channel.epoll.Epoll;
import io.netty.handler.ssl.OpenSsl;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;

/**
 * Reports the availability of the native libraries
 */
public final class NativeLibraries {

    private static final Logger logger = Logger.getLogger(NativeLibraries.class.getName());
    private static final AtomicBoolean reported = new AtomicBoolean();

    private static final boolean USE_EPOLL =
            !"false".equals(System.getProperty("org.xbib.helianthus.useEpoll", "true"));

    private static final boolean USE_OPENSSL =
            !"false".equals(System.getProperty("org.xbib.helianthus.useOpenSsl", "true"));

    private NativeLibraries() {
    }

    /**
     * Logs the availability of the native libraries. This method does nothing if it was
     * called once before.
     */
    public static void report() {
        if (!reported.compareAndSet(false, true)) {
            return;
        }

        if (USE_EPOLL) {
            logger.info("/dev/epoll: " +
                    (Epoll.isAvailable() ? "yes"
                            : "no (" + filterCause(Epoll.unavailabilityCause()) + ')'));
        } else {
            logger.info("/dev/epoll: disabled");
        }

        if (USE_OPENSSL) {
            logger.info("OpenSSL: " +
                    (OpenSsl.isAvailable() ? "yes (" + OpenSsl.versionString() + ", " +
                            OpenSsl.version() + ')'
                            : "no (" + filterCause(OpenSsl.unavailabilityCause()) + ')'));
        } else {
            logger.info("OpenSSL: disabled");
        }
    }

    private static Throwable filterCause(Throwable cause) {
        if (cause instanceof ExceptionInInitializerError) {
            return cause.getCause();
        }

        return cause;
    }

    /**
     * Returns {@code true} if JNI-based {@code /dev/epoll} transport is available.
     */
    public static boolean isEpollAvailable() {
        report();
        return USE_EPOLL && Epoll.isAvailable();
    }

    /**
     * Returns {@code true} if JNI-based OpenSSL/BoringSSL/LibreSSL transport is available.
     */
    public static boolean isOpenSslAvailable() {
        report();
        return USE_OPENSSL && OpenSsl.isAvailable();
    }
}
