package org.xbib.helianthus.common;

import static java.util.Objects.requireNonNull;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.EventLoop;
import io.netty.handler.codec.http.FullHttpResponse;
import io.netty.util.Attribute;
import io.netty.util.AttributeMap;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.ReferenceCounted;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.FutureListener;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.Promise;
import org.xbib.helianthus.common.logging.RequestLog;
import org.xbib.helianthus.common.logging.RequestLogBuilder;
import org.xbib.helianthus.common.logging.ResponseLog;
import org.xbib.helianthus.common.logging.ResponseLogBuilder;
import org.xbib.helianthus.common.util.Exceptions;

import java.text.MessageFormat;
import java.util.Iterator;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.logging.Logger;

/**
 * Provides information about an invocation and related utilities. Every remote invocation, regardless of if
 * it's client side or server side, has its own {@link RequestContext} instance.
 */
public interface RequestContext extends AttributeMap {

    /**
     * Returns the context of the invocation that is being handled in the current thread.
     *
     * @throws IllegalStateException if the context is unavailable in the current thread
     */
    static <T extends RequestContext> T current() {
        final T ctx = RequestContextThreadLocal.get();
        if (ctx == null) {
            throw new IllegalStateException(RequestContext.class.getSimpleName() + " unavailable");
        }
        return ctx;
    }

    /**
     * Maps the context of the invocation that is being handled in the current thread.
     *
     * @param mapper               the {@link Function} that maps the invocation
     * @param defaultValueSupplier the {@link Supplier} that provides the value when the context is unavailable
     *                             in the current thread. If {@code null}, the {@code null} will be returned
     *                             when the context is unavailable in the current thread.
     */
    static <T> T mapCurrent(
            Function<? super RequestContext, T> mapper, Supplier<T> defaultValueSupplier) {

        final RequestContext ctx = RequestContextThreadLocal.get();
        if (ctx != null) {
            return mapper.apply(ctx);
        }

        if (defaultValueSupplier != null) {
            return defaultValueSupplier.get();
        }

        return null;
    }

    /**
     * (Do not use; internal use only) Sets the invocation context of the current thread.
     */
    static PushHandle push(RequestContext ctx) {
        requireNonNull(ctx, "ctx");
        final RequestContext oldCtx = RequestContextThreadLocal.getAndSet(ctx);
        return oldCtx != null ? () -> RequestContextThreadLocal.set(oldCtx)
                : RequestContextThreadLocal::remove;
    }

    /**
     * Returns the {@link SessionProtocol} of this request.
     */
    SessionProtocol sessionProtocol();

    String method();

    /**
     * Returns the absolute path part of this invocation, as defined in
     * <a href="http://www.w3.org/Protocols/rfc2616/rfc2616-sec5.html#sec5.1.2">the
     * section 5.1.2 of RFC2616</a>.
     */
    String path();

    /**
     * Returns the request associated with this context.
     */
    <T> T request();

    RequestLogBuilder requestLogBuilder();

    ResponseLogBuilder responseLogBuilder();

    CompletableFuture<RequestLog> requestLogFuture();

    CompletableFuture<ResponseLog> responseLogFuture();

    Iterator<Attribute<?>> attrs();

    /**
     * Returns the {@link EventLoop} that is handling this invocation.
     */
    EventLoop eventLoop();

    /**
     * Returns an {@link EventLoop} that will make sure this invocation is set
     * as the current invocation before executing any callback. This should
     * almost always be used for executing asynchronous callbacks in service
     * code to make sure features that require the invocation context work
     * properly. Most asynchronous libraries like
     * {@link CompletableFuture} provide methods that
     * accept an {@link Executor} to run callbacks on.
     */
    default EventLoop contextAwareEventLoop() {
        return new RequestContextAwareEventLoop(this, eventLoop());
    }

    /**
     * Returns an {@link Executor} that will execute callbacks in the given
     * {@code executor}, making sure to propagate the current invocation context
     * into the callback execution. It is generally preferred to use
     * {@link #contextAwareEventLoop()} to ensure the callback stays on the
     * same thread as well.
     */
    default Executor makeContextAware(Executor executor) {
        return runnable -> executor.execute(makeContextAware(runnable));
    }

    /**
     * Returns a {@link Callable} that makes sure the current invocation context
     * is set and then invokes the input {@code callable}.
     */
    <T> Callable<T> makeContextAware(Callable<T> callable);

    /**
     * Returns a {@link Runnable} that makes sure the current invocation context
     * is set and then invokes the input {@code runnable}.
     */
    Runnable makeContextAware(Runnable runnable);

    /**
     * Returns a {@link FutureListener} that makes sure the current invocation
     * context is set and then invokes the input {@code listener}.
     */
    @Deprecated
    <T> FutureListener<T> makeContextAware(FutureListener<T> listener);

    /**
     * Returns a {@link ChannelFutureListener} that makes sure the current invocation
     * context is set and then invokes the input {@code listener}.
     */
    @Deprecated
    ChannelFutureListener makeContextAware(ChannelFutureListener listener);

    /**
     * Returns a {@link GenericFutureListener} that makes sure the current invocation
     * context is set and then invokes the input {@code listener}.
     */
    @Deprecated
    <T extends Future<?>> GenericFutureListener<T> makeContextAware(GenericFutureListener<T> listener);

    /**
     * Registers {@code callback} to be run when re-entering this {@link RequestContext},
     * usually when using the {@link #makeContextAware} family of methods. Any thread-local state
     * associated with this context should be restored by this callback.
     */
    void onEnter(Runnable callback);

    /**
     * Registers {@code callback} to be run when re-exiting this {@link RequestContext},
     * usually when using the {@link #makeContextAware} family of methods. Any thread-local state
     * associated with this context should be reset by this callback.
     */
    void onExit(Runnable callback);

    /**
     * Resolves the specified {@code promise} with the specified {@code result} so that the {@code promise} is
     * marked as 'done'. If {@code promise} is done already, this method does the following:
     * <ul>
     * <li>Log a warning about the failure, and</li>
     * <li>Release {@code result} if it is {@linkplain ReferenceCounted a reference-counted object},
     * such as {@link ByteBuf} and {@link FullHttpResponse}.</li>
     * </ul>
     * Note that a {@link Promise} can be done already even if you did not call this method in the following
     * cases:
     * <ul>
     * <li>Invocation timeout - The invocation associated with the {@link Promise} has been timed out.</li>
     * <li>User error - A service implementation called any of the following methods more than once:
     * <ul>
     * <li>{@link #resolvePromise(Promise, Object)}</li>
     * <li>{@link #rejectPromise(Promise, Throwable)}</li>
     * <li>{@link Promise#setSuccess(Object)}</li>
     * <li>{@link Promise#setFailure(Throwable)}</li>
     * <li>{@link Promise#cancel(boolean)}</li>
     * </ul>
     * </li>
     * </ul>
     */
    @Deprecated
    default void resolvePromise(Promise<?> promise, Object result) {
        @SuppressWarnings("unchecked")
        final Promise<Object> castPromise = (Promise<Object>) promise;

        if (castPromise.trySuccess(result)) {
            // Resolved successfully.
            return;
        }

        try {
            if (!(promise.cause() instanceof TimeoutException)) {
                // Log resolve failure unless it is due to a timeout.
                Logger.getLogger(RequestContext.class.getName()).warning(
                        MessageFormat.format("Failed to resolve a completed promise ({0}) with {1}", promise, result));
            }
        } finally {
            ReferenceCountUtil.safeRelease(result);
        }
    }

    /**
     * Rejects the specified {@code promise} with the specified {@code cause}. If {@code promise} is done
     * already, this method logs a warning about the failure. Note that a {@link Promise} can be done already
     * even if you did not call this method in the following cases:
     * <ul>
     * <li>Invocation timeout - The invocation associated with the {@link Promise} has been timed out.</li>
     * <li>User error - A service implementation called any of the following methods more than once:
     * <ul>
     * <li>{@link #resolvePromise(Promise, Object)}</li>
     * <li>{@link #rejectPromise(Promise, Throwable)}</li>
     * <li>{@link Promise#setSuccess(Object)}</li>
     * <li>{@link Promise#setFailure(Throwable)}</li>
     * <li>{@link Promise#cancel(boolean)}</li>
     * </ul>
     * </li>
     * </ul>
     */
    @Deprecated
    default void rejectPromise(Promise<?> promise, Throwable cause) {
        if (promise.tryFailure(cause)) {
            // Fulfilled successfully.
            return;
        }

        final Throwable firstCause = promise.cause();
        if (firstCause instanceof TimeoutException) {
            // Timed out already.
            return;
        }

        if (Exceptions.isExpected(cause)) {
            // The exception that was thrown after firstCause (often a transport-layer exception)
            // was a usual expected exception, not an error.
            return;
        }

        Logger.getLogger(RequestContext.class.getName()).warning(
                MessageFormat.format("Failed to reject a completed promise ({0}) with {1}", promise, cause, cause));
    }

    @FunctionalInterface
    interface PushHandle extends AutoCloseable {
        @Override
        void close();
    }
}
