package org.xbib.helianthus.common.logging;

import static java.util.Objects.requireNonNull;

import io.netty.util.Attribute;
import io.netty.util.AttributeKey;
import org.xbib.helianthus.common.util.CompletionActions;
import org.xbib.helianthus.common.util.ExceptionFormatter;
import org.xbib.helianthus.common.util.UnitFormatter;
import org.xbib.helianthus.internal.DefaultAttributeMap;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;

abstract class AbstractMessageLog<T extends MessageLog>
        extends CompletableFuture<T> implements MessageLog, MessageLogBuilder {

    private final DefaultAttributeMap attrs = new DefaultAttributeMap();
    private boolean startTimeNanosSet;
    private long startTimeNanos;
    private long contentLength;
    private long endTimeNanos;
    private Throwable cause;

    boolean start0() {
        if (isDone() || startTimeNanosSet) {
            return false;
        }

        startTimeNanos = System.nanoTime();
        startTimeNanosSet = true;
        return true;
    }

    @Override
    public long startTimeNanos() {
        return startTimeNanos;
    }

    @Override
    public void increaseContentLength(long deltaBytes) {
        if (deltaBytes < 0) {
            throw new IllegalArgumentException("deltaBytes: " + deltaBytes + " (expected: >= 0)");
        }
        if (isDone()) {
            return;
        }

        contentLength += deltaBytes;
    }

    @Override
    public void contentLength(long contentLength) {
        if (contentLength < 0) {
            throw new IllegalArgumentException("contentLength: " + contentLength + " (expected: >= 0)");
        }
        if (isDone()) {
            return;
        }

        this.contentLength = contentLength;
    }

    @Override
    public long contentLength() {
        return contentLength;
    }

    @Override
    public <V> Attribute<V> attr(AttributeKey<V> key) {
        return attrs.attr(key);
    }

    @Override
    public <V> boolean hasAttr(AttributeKey<V> key) {
        return attrs.hasAttr(key);
    }

    @Override
    public Iterator<Attribute<?>> attrs() {
        return attrs.attrs();
    }

    @Override
    public void end() {
        end0(null);
    }

    @Override
    public void end(Throwable cause) {
        requireNonNull(cause, "cause");
        end0(cause);
    }

    private void end0(Throwable cause) {
        if (isDone()) {
            return;
        }

        this.cause = cause;

        // Handle the case where end() was called without start()
        start0();

        final Iterator<Attribute<?>> attrs = attrs();
        if (attrs.hasNext()) {
            final List<CompletableFuture<?>> dependencies = new ArrayList<>(4);
            do {
                final Object a = attrs.next().get();
                if (a instanceof CompletableFuture) {
                    final CompletableFuture<?> f = (CompletableFuture<?>) a;
                    if (!f.isDone()) {
                        dependencies.add(f);
                    }
                }
            } while (attrs.hasNext());

            final CompletableFuture<?> future;
            switch (dependencies.size()) {
                case 0:
                    complete();
                    return;
                case 1:
                    future = dependencies.get(0);
                    break;
                default:
                    future = CompletableFuture.allOf(
                            dependencies.toArray(new CompletableFuture<?>[dependencies.size()]));
            }

            future.handle((unused1, unused2) -> complete())
                    .exceptionally(CompletionActions::log);
        } else {
            complete();
        }
    }

    private Void complete() {
        endTimeNanos = System.nanoTime();
        complete(self());
        return null;
    }

    @Override
    public long endTimeNanos() {
        return endTimeNanos;
    }

    @Override
    public Throwable cause() {
        return cause;
    }

    @SuppressWarnings("unchecked")
    private T self() {
        return (T) this;
    }

    @Override
    public final String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("timeSpan=")
                .append(startTimeNanos).append("+").append(UnitFormatter.elapsed(startTimeNanos, endTimeNanos))
                .append(",contentLength=")
                .append(UnitFormatter.size(contentLength));
        append(sb);
        sb.append(",cause=").append(ExceptionFormatter.format(cause))
                .append(",attrs=").append(attrs);
        return sb.toString();
    }

    protected abstract void append(StringBuilder sb);
}
