package org.xbib.helianthus.common.logging;

import org.xbib.helianthus.common.RequestContext;

import java.text.MessageFormat;
import java.util.logging.Level;
import java.util.logging.Logger;

@SuppressWarnings("MethodParameterNamingConvention")
public final class RequestContextAwareLogger {

    private final RequestContext ctx;
    private final Logger l;

    public RequestContextAwareLogger(RequestContext ctx, Logger l) {
        this.ctx = ctx;
        this.l = l;
    }

    private String decorate(String msg) {
        final String prefix = ctx.toString();
        return new StringBuilder(prefix.length() + 1 + msg.length())
                .append(prefix)
                .append(' ')
                .append(msg).toString();
    }

    public String getName() {
        return l.getName();
    }

    public boolean isTraceEnabled() {
        return l.isLoggable(Level.FINER);
    }

    public void trace(String msg) {
        if (isTraceEnabled()) {
            l.finer(decorate(msg));
        }
    }

    public void trace(String format, Object arg) {
        if (isTraceEnabled()) {
            l.finer(MessageFormat.format(decorate(format), arg));
        }
    }

    public void trace(String format, Object arg1, Object arg2) {
        if (isTraceEnabled()) {
            l.finer(MessageFormat.format(decorate(format), arg1, arg2));
        }
    }

    public void trace(String format, Object... arguments) {
        if (isTraceEnabled()) {
            l.finer(MessageFormat.format(decorate(format), arguments));
        }
    }

    public void trace(String msg, Throwable t) {
        if (isTraceEnabled()) {
            l.finer(MessageFormat.format(decorate(msg), t));
        }
    }

    public boolean isDebugEnabled() {
        return l.isLoggable(Level.FINE);
    }

    public void debug(String msg) {
        if (isDebugEnabled()) {
            l.fine(decorate(msg));
        }
    }

    public void debug(String format, Object arg) {
        if (isDebugEnabled()) {
            l.fine(MessageFormat.format(decorate(format), arg));
        }
    }

    public void debug(String format, Object arg1, Object arg2) {
        if (isDebugEnabled()) {
            l.fine(MessageFormat.format(decorate(format), arg1, arg2));
        }
    }

    public void debug(String format, Object... arguments) {
        if (isDebugEnabled()) {
            l.fine(MessageFormat.format(decorate(format), arguments));
        }
    }

    public void debug(String msg, Throwable t) {
        if (isDebugEnabled()) {
            l.fine(MessageFormat.format(decorate(msg), t));
        }
    }

    public boolean isInfoEnabled() {
        return l.isLoggable(Level.INFO);
    }

    public void info(String msg) {
        if (isInfoEnabled()) {
            l.info(decorate(msg));
        }
    }

    public void info(String format, Object arg) {
        if (isInfoEnabled()) {
            l.info(MessageFormat.format(decorate(format), arg));
        }
    }

    public void info(String format, Object arg1, Object arg2) {
        if (isInfoEnabled()) {
            l.info(MessageFormat.format(decorate(format), arg1, arg2));
        }
    }

    public void info(String format, Object... arguments) {
        if (isInfoEnabled()) {
            l.info(MessageFormat.format(decorate(format), arguments));
        }
    }

    public void info(String msg, Throwable t) {
        if (isInfoEnabled()) {
            l.info(MessageFormat.format(decorate(msg), t));
        }
    }

    public boolean isWarnEnabled() {
        return l.isLoggable(Level.WARNING);
    }

    public void warn(String msg) {
        if (isWarnEnabled()) {
            l.warning(decorate(msg));
        }
    }

    public void warn(String format, Object arg) {
        if (isWarnEnabled()) {
            l.warning(MessageFormat.format(decorate(format), arg));
        }
    }

    public void warn(String format, Object arg1, Object arg2) {
        if (isWarnEnabled()) {
            l.warning(MessageFormat.format(decorate(format), arg1, arg2));
        }
    }

    public void warn(String format, Object... arguments) {
        if (isWarnEnabled()) {
            l.warning(MessageFormat.format(decorate(format), arguments));
        }
    }

    public void warn(String msg, Throwable t) {
        if (isWarnEnabled()) {
            l.warning(MessageFormat.format(decorate(msg), t));
        }
    }

    public boolean isErrorEnabled() {
        return l.isLoggable(Level.SEVERE);
    }

    public void error(String msg) {
        if (isErrorEnabled()) {
            l.severe(decorate(msg));
        }
    }

    public void error(String format, Object arg) {
        if (isErrorEnabled()) {
            l.severe(MessageFormat.format(decorate(format), arg));
        }
    }

    public void error(String format, Object arg1, Object arg2) {
        if (isErrorEnabled()) {
            l.severe(MessageFormat.format(decorate(format), arg1, arg2));
        }
    }

    public void error(String format, Object... arguments) {
        if (isErrorEnabled()) {
            l.severe(MessageFormat.format(decorate(format), arguments));
        }
    }

    public void error(String msg, Throwable t) {
        if (isErrorEnabled()) {
            l.severe(MessageFormat.format(decorate(msg), t));
        }
    }

    @Override
    public String toString() {
        return "ServiceAwareLogger(" + l + ')';
    }
}
