package org.xbib.helianthus.common.util;

import static java.util.Objects.requireNonNull;

import io.netty.channel.Channel;
import io.netty.channel.ChannelException;
import io.netty.handler.codec.http2.Http2Exception;
import org.xbib.helianthus.common.ClosedSessionException;
import org.xbib.helianthus.common.SessionProtocol;

import java.io.IOException;
import java.nio.channels.ClosedChannelException;
import java.text.MessageFormat;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

/**
 * Provides the methods that are useful for handling exceptions.
 */
public final class Exceptions {

    private static final Logger logger = Logger.getLogger(Exceptions.class.getName());

    private static final Pattern IGNORABLE_SOCKET_ERROR_MESSAGE = Pattern.compile(
            "(?:connection.*(?:reset|closed|abort|broken)|broken.*pipe)", Pattern.CASE_INSENSITIVE);

    private static final Pattern IGNORABLE_HTTP2_ERROR_MESSAGE = Pattern.compile(
            "(?:stream closed)", Pattern.CASE_INSENSITIVE);

    private static final StackTraceElement[] EMPTY_STACK_TRACE = new StackTraceElement[0];

    private static final boolean VERBOSE =
            "true".equals(System.getProperty("org.xbib.helianthus.verboseExceptions", "false"));

    static {
        logger.info(MessageFormat.format("org.xbib.helianthus.verboseExceptions: {0}", VERBOSE));
    }

    private Exceptions() {
    }

    public static boolean isVerbose() {
        return VERBOSE;
    }

    /**
     * Logs the specified exception if it is {@linkplain #isExpected(Throwable)} unexpected}.
     */
    public static void logIfUnexpected(Logger logger, Channel ch, Throwable cause) {
        if (!logger.isLoggable(Level.WARNING) || isExpected(cause)) {
            return;
        }

        logger.log(Level.WARNING, MessageFormat.format("{0} Unexpected exception:", ch), cause);
    }

    /**
     * Logs the specified exception if it is {@linkplain #isExpected(Throwable)} unexpected}.
     */
    public static void logIfUnexpected(Logger logger, Channel ch, String debugData, Throwable cause) {

        if (!logger.isLoggable(Level.WARNING) || isExpected(cause)) {
            return;
        }

        logger.log(Level.WARNING, MessageFormat.format("{0} Unexpected exception: {1}", ch, debugData), cause);
    }

    /**
     * Logs the specified exception if it is {@linkplain #isExpected(Throwable)} unexpected}.
     */
    public static void logIfUnexpected(Logger logger, Channel ch, SessionProtocol protocol, Throwable cause) {
        if (!logger.isLoggable(Level.WARNING) || isExpected(cause)) {
            return;
        }

        logger.log(Level.WARNING, MessageFormat.format("{0}[{1}] Unexpected exception:", ch, protocolName(protocol)), cause);
    }

    /**
     * Logs the specified exception if it is {@linkplain #isExpected(Throwable)} unexpected}.
     */
    public static void logIfUnexpected(Logger logger, Channel ch, SessionProtocol protocol,
                                       String debugData, Throwable cause) {

        if (!logger.isLoggable(Level.WARNING) || isExpected(cause)) {
            return;
        }

        logger.log(Level.WARNING, MessageFormat.format("{0}[{1}] Unexpected exception: {2}", ch, protocolName(protocol), debugData), cause);
    }

    private static String protocolName(SessionProtocol protocol) {
        return protocol != null ? protocol.uriText() : "<unknown>";
    }

    /**
     * Returns {@code true} if the specified exception is expected to occur in well-known circumstances.
     * <ul>
     * <li>{@link ClosedChannelException}</li>
     * <li>{@link ClosedSessionException}</li>
     * <li>{@link IOException} - 'Connection reset/closed/aborted by peer'</li>
     * <li>'Broken pipe'</li>
     * <li>{@link Http2Exception} - 'Stream closed'</li>
     * </ul>
     */
    public static boolean isExpected(Throwable cause) {
        if (VERBOSE) {
            return true;
        }

        // We do not need to log every exception because some exceptions are expected to occur.

        if (cause instanceof ClosedChannelException || cause instanceof ClosedSessionException) {
            // Can happen when attempting to write to a channel closed by the other end.
            return true;
        }

        final String msg = cause.getMessage();
        if (msg != null) {
            if ((cause instanceof IOException || cause instanceof ChannelException) &&
                    IGNORABLE_SOCKET_ERROR_MESSAGE.matcher(msg).find()) {
                // Can happen when socket error occurs.
                return true;
            }

            if (cause instanceof Http2Exception && IGNORABLE_HTTP2_ERROR_MESSAGE.matcher(msg).find()) {
                // Can happen when disconnected prematurely.
                return true;
            }
        }

        return false;
    }

    /**
     * Empties the stack trace of the specified {@code exception}.
     */
    public static <T extends Throwable> T clearTrace(T exception) {
        requireNonNull(exception, "exception");
        exception.setStackTrace(EMPTY_STACK_TRACE);
        return exception;
    }
}
