package org.xbib.helianthus.internal.http;

import io.netty.buffer.ByteBuf;
import io.netty.channel.Channel;
import io.netty.handler.codec.http2.Http2Connection;
import io.netty.handler.codec.http2.Http2ConnectionAdapter;
import io.netty.handler.codec.http2.Http2Error;
import io.netty.handler.codec.http2.Http2Stream;

import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * A {@link Http2Connection.Listener} that logs the received GOAWAY frames and makes sure disconnection.
 */
public class Http2GoAwayListener extends Http2ConnectionAdapter {

    private static final Logger logger = Logger.getLogger(Http2GoAwayListener.class.getName());

    private final Channel ch;
    private boolean goAwaySent;

    public Http2GoAwayListener(Channel ch) {
        this.ch = ch;
    }

    private static String errorStr(long errorCode) {
        final Http2Error error = Http2Error.valueOf(errorCode);
        return error != null ? error.toString() + '(' + errorCode + ')'
                : "UNKNOWN(" + errorCode + ')';
    }

    @Override
    public void onGoAwaySent(int lastStreamId, long errorCode, ByteBuf debugData) {
        goAwaySent = true;
        onGoAway("Sent", lastStreamId, errorCode, debugData);
    }

    @Override
    public void onGoAwayReceived(int lastStreamId, long errorCode, ByteBuf debugData) {
        onGoAway("Received", lastStreamId, errorCode, debugData);

        // Send a GOAWAY back to the peer and close the connection gracefully if we did not send GOAWAY yet.
        // This will make sure that the connection is always closed after receiving GOAWAY,
        // because otherwise we have to wait until the peer who sent GOAWAY to us closes the connection.
        if (!goAwaySent) {
            ch.close();
        }
    }

    private void onGoAway(String sentOrReceived, int lastStreamId, long errorCode, ByteBuf debugData) {
        if (errorCode != Http2Error.NO_ERROR.code()) {
            if (logger.isLoggable(Level.WARNING)) {
                logger.warning(MessageFormat.format("{0} {1} a GOAWAY frame: lastStreamId={2}, errorCode={3}, debugData=\"{4}\"",
                        ch, sentOrReceived, lastStreamId, errorStr(errorCode),
                        debugData.toString(StandardCharsets.UTF_8)));
            }
        } else {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(MessageFormat.format("{0} {1} a GOAWAY frame: lastStreamId={2}, errorCode=NO_ERROR",
                        ch, sentOrReceived, lastStreamId));
            }
        }
    }

    @Override
    public void onStreamRemoved(Http2Stream stream) {
        if (stream.id() == 1) {
            logger.fine(MessageFormat.format("{0} HTTP/2 upgrade stream removed: {1}", ch, stream.state()));
        }
    }
}
