package org.xbib.helianthus.common;

import io.netty.channel.EventLoop;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import org.xbib.helianthus.common.util.SafeCloseable;

import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * A skeletal {@link RequestContext} implementation.
 */
public abstract class AbstractRequestContext implements RequestContext {

    @Override
    public final EventLoop contextAwareEventLoop() {
        return RequestContext.super.contextAwareEventLoop();
    }

    @Override
    public final Executor makeContextAware(Executor executor) {
        return RequestContext.super.makeContextAware(executor);
    }

    @SuppressWarnings("try")
    @Override
    public final <T> Callable<T> makeContextAware(Callable<T> callable) {
        return () -> {
            try (SafeCloseable ignored = propagateContextIfNotPresent()) {
                return callable.call();
            }
        };
    }

    @SuppressWarnings("try")
    @Override
    public final Runnable makeContextAware(Runnable runnable) {
        return () -> {
            try (SafeCloseable ignored = propagateContextIfNotPresent()) {
                runnable.run();
            }
        };
    }

    @SuppressWarnings("try")
    @Override
    public final <T, R> Function<T, R> makeContextAware(Function<T, R> function) {
        return t -> {
            try (SafeCloseable ignored = propagateContextIfNotPresent()) {
                return function.apply(t);
            }
        };
    }

    @SuppressWarnings("try")
    @Override
    public final <T, U, V> BiFunction<T, U, V> makeContextAware(BiFunction<T, U, V> function) {
        return (t, u) -> {
            try (SafeCloseable ignored = propagateContextIfNotPresent()) {
                return function.apply(t, u);
            }
        };
    }

    @SuppressWarnings("try")
    @Override
    public final <T> Consumer<T> makeContextAware(Consumer<T> action) {
        return t -> {
            try (SafeCloseable ignored = propagateContextIfNotPresent()) {
                action.accept(t);
            }
        };
    }

    @SuppressWarnings("try")
    @Override
    public final <T, U> BiConsumer<T, U> makeContextAware(BiConsumer<T, U> action) {
        return (t, u) -> {
            try (SafeCloseable ignored = propagateContextIfNotPresent()) {
                action.accept(t, u);
            }
        };
    }

    @Override
    public final <T extends Future<?>> GenericFutureListener<T> makeContextAware(GenericFutureListener<T> listener) {
        return future -> invokeOperationComplete(listener, future);
    }

    @SuppressWarnings("try")
    @Override
    public final <T> CompletionStage<T> makeContextAware(CompletionStage<T> stage) {
        final CompletableFuture<T> future = new RequestContextAwareCompletableFuture<>(this);
        stage.whenComplete((result, cause) -> {
            try (SafeCloseable ignored = propagateContextIfNotPresent()) {
                if (cause != null) {
                    future.completeExceptionally(cause);
                } else {
                    future.complete(result);
                }
            }
        });
        return future;
    }

    @Override
    public final <T> CompletableFuture<T> makeContextAware(CompletableFuture<T> future) {
        return RequestContext.super.makeContextAware(future);
    }

    @SuppressWarnings("try")
    private <T extends Future<?>> void invokeOperationComplete(GenericFutureListener<T> listener, T future)
            throws Exception {
        try (SafeCloseable ignored = propagateContextIfNotPresent()) {
            listener.operationComplete(future);
        }
    }

    private SafeCloseable propagateContextIfNotPresent() {
        return RequestContext.mapCurrent(currentContext -> {
            if (currentContext != this) {
                throw new IllegalStateException(
                        "Trying to call object made with makeContextAware or object on executor made with " +
                                "makeContextAware with context " + this +
                                ", but context is currently set to " + currentContext + ". This means the " +
                                "callback was passed from one invocation to another which is not allowed. Make " +
                                "sure you are not saving callbacks into shared state.");
            }
            return () -> { /* no-op */ };
        }, () -> RequestContext.push(this, true));
    }

    @Override
    public final int hashCode() {
        return super.hashCode();
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj);
    }
}
