package org.xbib.helianthus.common;

import static java.util.Objects.requireNonNull;

import org.xbib.helianthus.common.util.Exceptions;

import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;

public class AbstractRpcResponse extends CompletableFuture<Object> implements RpcResponse {

    private static final CancellationException CANCELLED = Exceptions.clearTrace(new CancellationException());

    private static final AtomicReferenceFieldUpdater<AbstractRpcResponse, Throwable> causeUpdater =
            AtomicReferenceFieldUpdater.newUpdater(AbstractRpcResponse.class, Throwable.class, "cause");

    private volatile Throwable cause;

    protected AbstractRpcResponse() {
    }

    protected AbstractRpcResponse(Object result) {
        complete(result);
    }

    protected AbstractRpcResponse(Throwable cause) {
        requireNonNull(cause, "cause");
        completeExceptionally(cause);
    }

    @Override
    public final Throwable getCause() {
        return cause;
    }

    @Override
    public boolean completeExceptionally(Throwable cause) {
        causeUpdater.compareAndSet(this, null, requireNonNull(cause));
        return super.completeExceptionally(cause);
    }

    @Override
    public void obtrudeException(Throwable cause) {
        this.cause = requireNonNull(cause);
        super.obtrudeException(cause);
    }

    @Override
    public boolean cancel(boolean mayInterruptIfRunning) {
        return completeExceptionally(CANCELLED) || isCancelled();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        if (isDone()) {
            if (isCompletedExceptionally()) {
                sb.append("AbstractRpcResponse:")
                        .append("cause=")
                        .append(cause);
                return sb.toString();
            } else {
                sb.append("AbstractRpcResponse:")
                        .append("value=")
                        .append(getNow(null));
                return sb.toString();
            }
        }

        final int count = getNumberOfDependents();
        if (count == 0) {
            sb.append("not completed");
            return sb.toString();
        } else {
            sb.append("not completed, dependents=")
                    .append(count);
            return sb.toString();
        }
    }
}
