package org.xbib.helianthus.internal;

import io.netty.channel.Channel;
import io.netty.channel.ChannelConfig;

import java.util.concurrent.atomic.AtomicInteger;

public final class InboundTrafficController extends AtomicInteger {

    private static final long serialVersionUID = 420503276551000218L;

    private static int numDeferredReads;
    private final ChannelConfig cfg;
    private final int highWatermark;
    private final int lowWatermark;
    private volatile boolean suspended;

    public InboundTrafficController(Channel channel) {
        this(channel, 128 * 1024, 64 * 1024);
    }

    public InboundTrafficController(Channel channel, int highWatermark, int lowWatermark) {
        cfg = channel != null ? channel.config() : null;
        this.highWatermark = highWatermark;
        this.lowWatermark = lowWatermark;
    }

    public static int numDeferredReads() {
        return numDeferredReads;
    }

    public void inc(int numProducedBytes) {
        final int oldValue = getAndAdd(numProducedBytes);
        if (oldValue <= highWatermark && oldValue + numProducedBytes > highWatermark && cfg != null) {
            cfg.setAutoRead(false);
            numDeferredReads++;
            suspended = true;
        }
    }

    public void dec(int numConsumedBytes) {
        final int oldValue = getAndAdd(-numConsumedBytes);
        if (oldValue > lowWatermark && oldValue - numConsumedBytes <= lowWatermark && cfg != null) {
            cfg.setAutoRead(true);
            suspended = false;
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("InboundTrafficController:suspended=")
                .append(suspended)
                .append(",unconsumed=")
                .append(get())
                .append(",watermarks=")
                .append(highWatermark + "/" + lowWatermark);
        return sb.toString();
    }
}
