package org.xbib.helianthus.common;

import io.netty.channel.EventLoop;
import io.netty.util.Attribute;
import io.netty.util.AttributeMap;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import org.xbib.helianthus.common.logging.RequestLog;
import org.xbib.helianthus.common.logging.RequestLogBuilder;
import org.xbib.helianthus.common.util.SafeCloseable;

import javax.net.ssl.SSLSession;
import java.net.SocketAddress;
import java.util.Iterator;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.logging.Logger;

/**
 * Provides information about a {@link Request}, its {@link Response} and related utilities.
 */
public interface RequestContext extends AttributeMap {

    Logger logger = Logger.getLogger(RequestContext.class.getName());

    /**
     * Returns the context of the {@link Request} that is being handled in the current thread.
     *
     * @throws IllegalStateException if the context is unavailable in the current thread
     */
    static <T extends RequestContext> T current() {
        final T ctx = RequestContextThreadLocal.get();
        if (ctx == null) {
            throw new IllegalStateException(RequestContext.class.getSimpleName() + " unavailable");
        }
        return ctx;
    }

    /**
     * Maps the context of the {@link Request} that is being handled in the current thread.
     *
     * @param mapper the {@link Function} that maps the {@link RequestContext}
     * @param defaultValueSupplier the {@link Supplier} that provides the value when the context is unavailable
     *                             in the current thread. If {@code null}, the {@code null} will be returned
     *                             when the context is unavailable in the current thread.
     */
    static <T> T mapCurrent(
            Function<? super RequestContext, T> mapper, Supplier<T> defaultValueSupplier) {

        final RequestContext ctx = RequestContextThreadLocal.get();
        if (ctx != null) {
            return mapper.apply(ctx);
        }

        if (defaultValueSupplier != null) {
            return defaultValueSupplier.get();
        }

        return null;
    }

    /**
     * Pushes the specified context to the thread-local stack. To pop the context from the stack, call
     * {@link SafeCloseable#close()}, which can be done using a {@code try-finally} block:
     * <pre>{@code
     * try (SafeCloseable ignored = RequestContext.push(ctx)) {
     *     ...
     * }
     * }</pre>
     *
     * <p>The callbacks added by {@link #onEnter(Runnable)} and {@link #onExit(Runnable)} will be invoked
     * when the context is pushed to and removed from the thread-local stack respectively.
     *
     * <p>NOTE: In case of re-entrance, the callbacks will never run.
     */
    static SafeCloseable push(RequestContext ctx) {
        return push(ctx, true);
    }

    /**
     * Pushes the specified context to the thread-local stack. To pop the context from the stack, call
     * {@link SafeCloseable#close()}, which can be done using a {@code try-finally} block:
     * <pre>{@code
     * try (PushHandle ignored = RequestContext.push(ctx, true)) {
     *     ...
     * }
     * }</pre>
     *
     * <p>NOTE: This method is only useful when it is undesirable to invoke the callbacks, such as replacing
     *          the current context with another. Prefer {@link #push(RequestContext)} otherwise.
     *
     * @param runCallbacks if {@code true}, the callbacks added by {@link #onEnter(Runnable)} and
     *                     {@link #onExit(Runnable)} will be invoked when the context is pushed to and
     *                     removed from the thread-local stack respectively.
     *                     If {@code false}, no callbacks will be executed.
     *                     NOTE: In case of re-entrance, the callbacks will never run.
     */
    static SafeCloseable push(RequestContext ctx, boolean runCallbacks) {
        final RequestContext oldCtx = RequestContextThreadLocal.getAndSet(ctx);
        if (oldCtx == ctx) {
            // Reentrance
            return () -> { /* no-op */ };
        }

        if (runCallbacks) {
            ctx.invokeOnEnterCallbacks();
            if (oldCtx != null) {
                return () -> {
                    ctx.invokeOnExitCallbacks();
                    RequestContextThreadLocal.set(oldCtx);
                };
            } else {
                return () -> {
                    ctx.invokeOnExitCallbacks();
                    RequestContextThreadLocal.remove();
                };
            }
        } else {
            if (oldCtx != null) {
                return () -> RequestContextThreadLocal.set(oldCtx);
            } else {
                return RequestContextThreadLocal::remove;
            }
        }
    }

    /**
     * Returns the {@link SessionProtocol} of the current {@link Request}.
     */
    SessionProtocol sessionProtocol();

    /**
     * Returns the remote address of this request, or {@code null} if the connection is not established yet.
     */
    <A extends SocketAddress> A remoteAddress();

    /**
     * Returns the local address of this request, or {@code null} if the connection is not established yet.
     */
    <A extends SocketAddress> A localAddress();

    /**
     * The {@link SSLSession} for this request if the connection is made over TLS, or {@code null} if
     * the connection is not established yet or the connection is not a TLS connection.
     */
    SSLSession sslSession();

    /**
     * Returns the session-layer method name of the current {@link Request}. e.g. "GET" or "POST" for HTTP
     */
    String method();

    /**
     * Returns the absolute path part of the current {@link Request}, as defined in
     * <a href="http://www.w3.org/Protocols/rfc2616/rfc2616-sec5.html#sec5.1.2">the
     * section 5.1.2 of RFC2616</a>.
     */
    String path();

    /**
     * Returns the {@link Request} associated with this context.
     */
    <T> T request();

    /**
     * Returns the {@link RequestLog} that contains the information about the current {@link Request}.
     */
    RequestLog log();

    /**
     * Returns the {@link RequestLogBuilder} that collects the information about the current {@link Request}.
     */
    RequestLogBuilder logBuilder();

    /**
     * Returns all {@link Attribute}s set in this context.
     */
    Iterator<Attribute<?>> attrs();

    /**
     * Returns the {@link EventLoop} that is handling the current {@link Request}.
     */
    EventLoop eventLoop();

    /**
     * Returns an {@link EventLoop} that will make sure this {@link RequestContext} is set as the current
     * context before executing any callback. This should almost always be used for executing asynchronous
     * callbacks in service code to make sure features that require the {@link RequestContext} work properly.
     * Most asynchronous libraries like {@link CompletableFuture} provide methods that accept an
     * {@link Executor} to run callbacks on.
     */
    default EventLoop contextAwareEventLoop() {
        return new RequestContextAwareEventLoop(this, eventLoop());
    }

    /**
     * Returns an {@link Executor} that will execute callbacks in the given {@code executor}, making sure to
     * propagate the current {@link RequestContext} into the callback execution. It is generally preferred to
     * use {@link #contextAwareEventLoop()} to ensure the callback stays on the same thread as well.
     */
    @SuppressWarnings("overloads")
    default Executor makeContextAware(Executor executor) {
        return runnable -> executor.execute(makeContextAware(runnable));
    }

    /**
     * Returns an {@link ExecutorService} that will execute callbacks in the given {@code executor}, making
     * sure to propagate the current {@link RequestContext} into the callback execution.
     */
    default ExecutorService makeContextAware(ExecutorService executor) {
        return new RequestContextAwareExecutorService(this, executor);
    }

    /**
     * Returns a {@link Callable} that makes sure the current {@link RequestContext} is set and then invokes
     * the input {@code callable}.
     */
    <T> Callable<T> makeContextAware(Callable<T> callable);

    /**
     * Returns a {@link Runnable} that makes sure the current {@link RequestContext} is set and then invokes
     * the input {@code runnable}.
     */
    Runnable makeContextAware(Runnable runnable);

    /**
     * Returns a {@link Function} that makes sure the current {@link RequestContext} is set and then invokes
     * the input {@code function}.
     */
    @SuppressWarnings("overloads")
    <T, R> Function<T, R> makeContextAware(Function<T, R> function);

    /**
     * Returns a {@link BiFunction} that makes sure the current {@link RequestContext} is set and then invokes
     * the input {@code function}.
     */
    @SuppressWarnings("overloads")
    <T, U, V> BiFunction<T, U, V> makeContextAware(BiFunction<T, U, V> function);

    /**
     * Returns a {@link Consumer} that makes sure the current {@link RequestContext} is set and then invokes
     * the input {@code action}.
     */
    @SuppressWarnings("overloads")
    <T> Consumer<T> makeContextAware(Consumer<T> action);

    /**
     * Returns a {@link BiConsumer} that makes sure the current {@link RequestContext} is set and then invokes
     * the input {@code action}.
     */
    @SuppressWarnings("overloads")
    <T, U> BiConsumer<T, U> makeContextAware(BiConsumer<T, U> action);

    /**
     * Returns a {@link GenericFutureListener} that makes sure the current {@link RequestContext} is set and
     * then invokes the input {@code listener}.
     */
    @SuppressWarnings("overloads")
    <T extends Future<?>> GenericFutureListener<T> makeContextAware(GenericFutureListener<T> listener);

    /**
     * Returns a {@link CompletionStage} that makes sure the current {@link CompletionStage} is set and
     * then invokes the input {@code stage}.
     */
    <T> CompletionStage<T> makeContextAware(CompletionStage<T> stage);

    /**
     * Returns a {@link CompletableFuture} that makes sure the current {@link CompletableFuture} is set and
     * then invokes the input {@code future}.
     */
    default <T> CompletableFuture<T> makeContextAware(CompletableFuture<T> future) {
        return makeContextAware((CompletionStage<T>) future).toCompletableFuture();
    }

    /**
     * Registers {@code callback} to be run when re-entering this {@link RequestContext}, usually when using
     * the {@link #makeContextAware} family of methods. Any thread-local state associated with this context
     * should be restored by this callback.
     */
    void onEnter(Runnable callback);

    /**
     * Registers {@code callback} to be run when re-exiting this {@link RequestContext}, usually when using
     * the {@link #makeContextAware} family of methods. Any thread-local state associated with this context
     * should be reset by this callback.
     */
    void onExit(Runnable callback);

    /**
     * Invokes all {@link #onEnter(Runnable)} callbacks. It is discouraged to use this method directly.
     * Use {@link #makeContextAware(Runnable)} or {@link #push(RequestContext, boolean)} instead so that
     * the callbacks are invoked automatically.
     */
    void invokeOnEnterCallbacks();

    /**
     * Invokes all {@link #onExit(Runnable)} callbacks. It is discouraged to use this method directly.
     * Use {@link #makeContextAware(Runnable)} or {@link #push(RequestContext, boolean)} instead so that
     * the callbacks are invoked automatically.
     */
    void invokeOnExitCallbacks();
}
