package org.xbib.helianthus.common;

import java.util.Locale;
import java.util.Optional;

/**
 * Serialization format of a remote procedure call and its reply.
 */
public enum SerializationFormat {

    /**
     * No serialization format. Used when no serialization/deserialization is desired.
     */
    NONE("none", "application", "x-none"),

    /**
     * Unknown serialization format. Used when some serialization format is desired but the server
     * failed to understand/recognize it.
     */
    UNKNOWN("unknown", "application", "x-unknown");

    private final String uriText;
    private final String mediaType;

    SerializationFormat(String uriText, String mediaType, String subType) {
        this.uriText = uriText;
        this.mediaType = mediaType + "/" + subType;

    }

    /**
     * Returns the serialization format corresponding to the passed in {@code mediaType}, or
     * {@link Optional#empty} if the media type is not recognized. {@code null} is treated as an unknown
     * mimetype.
     */
    public static Optional<SerializationFormat> fromMediaType(String mediaType) {
        if (mediaType == null || mediaType.isEmpty()) {
            return Optional.empty();
        }

        final int semicolonIdx = mediaType.indexOf(';');
        final String paramPart;
        if (semicolonIdx >= 0) {
            paramPart = mediaType.substring(semicolonIdx).toLowerCase(Locale.US);
            mediaType = mediaType.substring(0, semicolonIdx).toLowerCase(Locale.US).trim();
        } else {
            paramPart = null;
            mediaType = mediaType.toLowerCase(Locale.US).trim();
        }

        if (NONE.mediaType().equals(mediaType)) {
            return Optional.of(NONE);
        }

        return Optional.empty();
    }

    /**
     * Returns the textual representation of this format for use in a {@link Scheme}.
     */
    public String uriText() {
        return uriText;
    }

    public String mediaType() {
        return mediaType;
    }
}
