package org.xbib.helianthus.common.http;

import static java.util.Objects.requireNonNull;

import org.xbib.helianthus.common.util.ImmutableList;

import java.util.Collections;
import java.util.List;

/**
 * A complete HTTP message whose content is readily available as a single {@link HttpData}. It can be an
 * HTTP request or an HTTP response depending on what header values it contains. For example, having a
 * {@link HttpHeaderNames#STATUS} header could mean it is an HTTP response.
 */
public interface AggregatedHttpMessage {

    /**
     * Creates a new HTTP request with empty content.
     *
     * @param method the HTTP method of the request
     * @param path the path of the request
     * @return aggregated HTTP message
     */
    static AggregatedHttpMessage of(HttpMethod method, String path) {
        return of(HttpHeaders.of(method, path));
    }

    /**
     * Creates a new HTTP request.
     *
     * @param method the HTTP method of the request
     * @param path the path of the request
     * @param content the content of the request
     * @return aggregated HTTP message
     */
    static AggregatedHttpMessage of(HttpMethod method, String path, HttpData content) {
        return of(HttpHeaders.of(method, path), content);
    }

    /**
     * Creates a new HTTP response with empty content.
     *
     * @param statusCode the HTTP status code
     * @return aggregated HTTP message
     */
    static AggregatedHttpMessage of(int statusCode) {
        return of(HttpStatus.valueOf(statusCode));
    }

    /**
     * Creates a new HTTP response with empty content.
     *
     * @param status the HTTP status
     * @return aggregated HTTP message
     */
    static AggregatedHttpMessage of(HttpStatus status) {
        return of(HttpHeaders.of(status));
    }

    /**
     * Creates a new HTTP response.
     *
     * @param status the HTTP status
     * @param content the content of the HTTP response
     * @return aggregated HTTP message
     */
    static AggregatedHttpMessage of(HttpStatus status, HttpData content) {
        return of(HttpHeaders.of(status), content);
    }


    static AggregatedHttpMessage of(HttpHeaders headers, HttpStatus status, HttpData content) {
        return of(headers.status(status), content);
    }

    /**
     * Creates a new HTTP message with empty content.
     *
     * @param headers the HTTP headers
     * @return aggregated HTTP message
     */
    static AggregatedHttpMessage of(HttpHeaders headers) {
        return of(headers, HttpData.EMPTY_DATA, HttpHeaders.EMPTY_HEADERS, null);
    }

    /**
     * Creates a new HTTP message.
     *
     * @param headers the HTTP headers
     * @param content the content of the HTTP message
     * @return aggregated HTTP message
     */
    static AggregatedHttpMessage of(HttpHeaders headers, HttpData content) {
        return of(headers, content, HttpHeaders.EMPTY_HEADERS, null);
    }

    /**
     * Creates a new HTTP message.
     *
     * @param headers the HTTP headers
     * @param content the content of the HTTP message
     * @param trailingHeaders the trailing HTTP headers
     * @return aggregated HTTP message
     */
    static AggregatedHttpMessage of(HttpHeaders headers, HttpData content, HttpHeaders trailingHeaders, String followUrl) {
        return of(Collections.emptyList(), headers, content, trailingHeaders, followUrl);
    }

    static AggregatedHttpMessage of(Iterable<HttpHeaders> informationals, HttpHeaders headers,
                                    HttpData content, HttpHeaders trailingHeaders, String followUrl) {

        requireNonNull(informationals, "informationals");
        requireNonNull(headers, "headers");
        requireNonNull(content, "content");
        requireNonNull(trailingHeaders, "trailingHeaders");

        return new DefaultAggregatedHttpMessage(ImmutableList.copyOf(informationals),
                headers, content, trailingHeaders, followUrl);
    }

    /**
     * Returns the informational class (1xx) HTTP headers.
     * @return list og HTTP headers
     */
    List<HttpHeaders> informationals();

    /**
     * Returns the HTTP headers.
     */
    HttpHeaders headers();

    /**
     * Returns the trailing HTTP headers.
     */
    HttpHeaders trailingHeaders();

    /**
     * Returns the content of this message.
     */
    HttpData content();

    /**
     * Returns the {@link HttpHeaderNames#SCHEME SCHEME} of this message.
     *
     * @return the scheme, or {@code null} if there's no such header
     */
    default String scheme() {
        return headers().scheme();
    }

    /**
     * Returns the {@link HttpHeaderNames#METHOD METHOD} of this message.
     *
     * @return the method, or {@code null} if there's no such header
     */
    default HttpMethod method() {
        return headers().method();
    }

    /**
     * Returns the {@link HttpHeaderNames#PATH PATH} of this message.
     *
     * @return the path, or {@code null} if there's no such header
     */
    default String path() {
        return headers().path();
    }

    /**
     * Returns the {@link HttpHeaderNames#AUTHORITY AUTHORITY} of this message, in the form of
     * {@code "hostname:port"}.
     *
     * @return the authority, or {@code null} if there's no such header
     */
    default String authority() {
        return headers().authority();
    }

    /**
     * Returns the {@link HttpHeaderNames#STATUS STATUS} of this message.
     *
     * @return the status, or {@code null} if there's no such header
     */
    default HttpStatus status() {
        return headers().status();
    }

    String followUrl();
}
