package org.xbib.helianthus.common.logging;

import io.netty.channel.Channel;
import org.xbib.helianthus.common.Request;
import org.xbib.helianthus.common.RequestContext;
import org.xbib.helianthus.common.Response;
import org.xbib.helianthus.common.RpcRequest;
import org.xbib.helianthus.common.RpcResponse;
import org.xbib.helianthus.common.Scheme;
import org.xbib.helianthus.common.SerializationFormat;
import org.xbib.helianthus.common.SessionProtocol;
import org.xbib.helianthus.common.http.HttpHeaders;

import java.util.Arrays;
import java.util.Set;

import static java.util.Objects.requireNonNull;

/**
 * A set of informational properties collected while processing a {@link Request} and its {@link Response}.
 * The properties provided by this class are not always fully available. Check the availability of each
 * property using {@link #isAvailable(RequestLogAvailability)} or {@link #availabilities()}. Attempting to
 * access the properties that are not available yet will cause a {@link RequestLogAvailabilityException}.
 * Use {@link #addListener(RequestLogListener, RequestLogAvailability)} to get notified when the interested
 * properties are available.
 *
 * @see RequestContext#log()
 * @see RequestLogAvailability
 * @see RequestLogListener
 */
public interface RequestLog {

    /**
     * Returns the set of satisfied {@link RequestLogAvailability}s.
     * @return set of request log availabilities
     */
    Set<RequestLogAvailability> availabilities();

    /**
     * Returns {@code true} if the specified {@link RequestLogAvailability} is satisfied.
     * @param availability availability
     * @return true if available, false if not
     */
    boolean isAvailable(RequestLogAvailability availability);

    /**
     * Returns {@code true} if all of the specified {@link RequestLogAvailability}s are satisfied.
     * @param availabilities availabilities
     * @return true if available
     */
    default boolean isAvailable(RequestLogAvailability... availabilities) {
        for (RequestLogAvailability k : requireNonNull(availabilities, "availabilities")) {
            if (!isAvailable(k)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Returns {@code true} if all of the specified {@link RequestLogAvailability}s are satisfied.
     * @param availabilities availabilities
     * @return true if available
     */
    default boolean isAvailable(Iterable<RequestLogAvailability> availabilities) {
        for (RequestLogAvailability k : requireNonNull(availabilities, "availabilities")) {
            if (!isAvailable(k)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Ensures that the specified {@link RequestLogAvailability} is satisfied.
     * @param availability availability
     * @throws RequestLogAvailabilityException if not satisfied yet
     */
    default void ensureAvailability(RequestLogAvailability availability) {
        if (!isAvailable(availability)) {
            throw new RequestLogAvailabilityException(availability.name());
        }
    }

    /**
     * Ensures that all of the specified {@link RequestLogAvailability}s are satisfied.
     * @param availabilities availabilities
     * @throws RequestLogAvailabilityException if not satisfied yet
     */
    default void ensureAvailability(RequestLogAvailability... availabilities) {
        if (!isAvailable(availabilities)) {
            throw new RequestLogAvailabilityException(Arrays.toString(availabilities));
        }
    }

    /**
     * Ensures that all of the specified {@link RequestLogAvailability}s are satisfied.
     * @param properties properties
     * @throws RequestLogAvailabilityException if not satisfied yet
     */
    default void ensureAvailability(Iterable<RequestLogAvailability> properties) {
        if (!isAvailable(properties)) {
            throw new RequestLogAvailabilityException(properties.toString());
        }
    }

    /**
     * Adds the specified {@link RequestLogListener} so that it's notified when the specified
     * {@link RequestLogAvailability} is satisfied.
     * @param listener listener
     * @param availability availability
     */
    void addListener(RequestLogListener listener, RequestLogAvailability availability);

    /**
     * Adds the specified {@link RequestLogListener} so that it's notified when all of the specified
     * {@link RequestLogAvailability}s are satisfied.
     * @param listener listener
     * @param availabilities availabilities
     */
    void addListener(RequestLogListener listener, RequestLogAvailability... availabilities);

    /**
     * Adds the specified {@link RequestLogListener} so that it's notified when all of the specified
     * {@link RequestLogAvailability}s are satisfied.
     * @param listener listener
     * @param availabilities availabilities
     */
    void addListener(RequestLogListener listener, Iterable<RequestLogAvailability> availabilities);

    /**
     * Returns the {@link RequestContext} associated with the {@link Request} being handled.
     * This method returns non-{@code null} regardless the current {@link RequestLogAvailability}.
     * @return request c
     */
    RequestContext context();

    /**
     * Returns the time when the processing of the request started, in millis since the epoch.
     * @return time in millis
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    long requestStartTimeMillis();

    /**
     * Returns the duration that was taken to consume or produce the request completely, in nanoseconds.
     * @return duration in nanos
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    long requestDurationNanos();

    /**
     * Returns the length of the request content.
     * @return length in bytes
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    long requestLength();

    /**
     * Returns the cause of request processing failure.
     *
     * @return the cause. {@code null} if the request was processed completely.
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    Throwable requestCause();

    /**
     * Returns the time when the processing of the response started, in millis since the epoch.
     * @return time in millis
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    long responseStartTimeMillis();

    /**
     * Returns the duration that was taken to consume or produce the response completely, in nanoseconds.
     * @return duration in nanos
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    long responseDurationNanos();

    /**
     * Returns the length of the response content.
     * @return length in bytes
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    long responseLength();

    /**
     * Returns the cause of response processing failure.
     *
     * @return the cause. {@code null} if the response was processed completely.
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    Throwable responseCause();

    /**
     * Returns the amount of time taken since the {@link Request} processing started and until the
     * {@link Response} processing ended. This property is available only when both
     * {@link RequestLogAvailability#REQUEST_START} and {@link RequestLogAvailability#RESPONSE_END} are
     * available.
     * @return duration nanos
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    long totalDurationNanos();

    /**
     * Returns the Netty {@link Channel} which handled the {@link Request}.
     * @return channel
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    Channel channel();

    /**
     * Returns the {@link SessionProtocol} of the {@link Request}.
     * @return session protocol
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    SessionProtocol sessionProtocol();

    /**
     * Returns the {@link SerializationFormat} of the {@link Request}.
     * @return serialization
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    SerializationFormat serializationFormat();

    /**
     * Returns the {@link Scheme} of the {@link Request}.
     * @return scheme
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    Scheme scheme();

    /**
     * Returns the host name of the {@link Request}.
     * @return host
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    String host();

    /**
     * Returns the method of the {@link Request}.
     * @return method
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    String method();

    /**
     * Returns the path of the {@link Request}.
     * @return path
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    String path();

    /**
     * Returns the status code specific to the {@link Response} of the current {@link SessionProtocol}.
     * @return status code
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    int statusCode();

    /**
     * Returns the {@link SessionProtocol}-level envelope object of the {@link Request}.
     *
     * @return {@link HttpHeaders} for HTTP, or {@code null} for others
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    Object requestEnvelope();

    /**
     * Returns the high-level content object of the {@link Request}, which is specific
     * to the {@link SerializationFormat}.
     *
     * @return {@link RpcRequest} for RPC, or {@code null} for others
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    Object requestContent();

    /**
     * Returns the low-level content object of the {@link Request}, which is specific
     * to the {@link SerializationFormat}.
     *
     * @return {@code ThriftCall} for Thrift, or {@code null} for others
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    Object rawRequestContent();

    /**
     * Returns the {@link SessionProtocol}-level envelope object of the {@link Response}.
     *
     * @return {@link HttpHeaders} for HTTP, or {@code null} for others
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    Object responseEnvelope();

    /**
     * Returns the high-level content object of the {@link Response}, which is specific
     * to the {@link SerializationFormat}.
     *
     * @return {@link RpcResponse} for RPC, or {@code null} for others
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    Object responseContent();

    /**
     * Returns the low-level content object of the {@link Response}, which is specific
     * to the {@link SerializationFormat}.
     *
     * @return {@code ThriftReply} for Thrift, or {@code null} for others
     * @throws RequestLogAvailabilityException if this property is not available yet
     */
    Object rawResponseContent();

    /**
     * Returns the string representation of the {@link Request}.
     * @return string representation
     */
    String toStringRequestOnly();

    /**
     * Returns the string representation of the {@link Response}.
     * @return string representation
     */
    String toStringResponseOnly();
}
