package org.xbib.helianthus.common.logging;

import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * A special {@link Set} implementation that pre-populates all possible combinations of
 * {@link RequestLogAvailability}. Note that the number of all combinations are very small, because most
 * combinations can be merged. e.g. [ REQUEST_START, SCHEME ] and [ SCHEME ] are same because their
 * getterFlags are identical.
 */
final class RequestLogAvailabilitySet extends AbstractSet<RequestLogAvailability> {

    private static final Map<Integer, RequestLogAvailabilitySet> map = new HashMap<>();

    static {
        // Pre-populate all possible instances.
        final RequestLogAvailability[] values = RequestLogAvailability.values();
        final int end = (int)Math.pow(2, values.length);
        for (int i = 0; i < end; i++) {
            int flags = 0;
            for (RequestLogAvailability v : values) {
                if ((i & 1 << v.ordinal()) != 0) {
                    flags |= v.getterFlags();
                }
            }

            if (map.containsKey(flags)) {
                continue;
            }

            map.put(flags, new RequestLogAvailabilitySet(flags));
        }
    }

    static RequestLogAvailabilitySet of(int flags) {
        final RequestLogAvailabilitySet availabilities = map.get(flags);
        assert availabilities != null;
        return availabilities;
    }

    private final int flags;
    private final RequestLogAvailability[] values;

    private RequestLogAvailabilitySet(int flags) {
        this.flags = flags;

        final List<RequestLogAvailability> values = new ArrayList<>();
        for (RequestLogAvailability v : RequestLogAvailability.values()) {
            if ((flags & v.getterFlags()) == flags) {
                values.add(v);
            }
        }
        this.values = values.toArray(new RequestLogAvailability[values.size()]);
    }


    @Override
    public Iterator<RequestLogAvailability> iterator() {
        return Arrays.asList(values).iterator();
    }

    @Override
    public int size() {
        return values.length;
    }

    @Override
    public boolean contains(Object e) {
        if (!(e instanceof RequestLogAvailability)) {
            return false;
        }

        final int flags = ((RequestLogAvailability) e).getterFlags();
        return (this.flags & flags) == flags;
    }

    @Override
    public boolean equals(Object o) {
        return this == o;
    }

    @Override
    public int hashCode() {
        return System.identityHashCode(this);
    }

    @Override
    public String toString() {
        return Arrays.toString(values);
    }
}
