package org.xbib.interlibrary.z.hbz;

import org.xbib.content.settings.Settings;
import org.xbib.interlibrary.action.avail.AvailRequest;
import org.xbib.interlibrary.action.avail.AvailResponse;
import org.xbib.interlibrary.api.Library;
import org.xbib.interlibrary.api.service.ServiceArguments;
import org.xbib.interlibrary.common.util.MultiMap;
import org.xbib.interlibrary.z.AbstractZAvailService;
import org.xbib.interlibrary.z.ZClientHelper;
import org.xbib.marc.Marc;
import org.xbib.marc.MarcField;
import org.xbib.marc.MarcRecord;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 */
public class AvailService extends AbstractZAvailService {

    private static final Logger logger = Logger.getLogger(AvailService.class.getName());

    private final Map<String, Object> country;

    private final Map<String, Object> language;

    private final Map<String, Object> monographcodes;

    private final Map<String, Object> serialcodes;

    private final Map<String, Object> statuscodes;

    protected AvailService(Settings settings,
                           ServiceArguments arguments,
                           ZClientHelper client) {
        super(settings, arguments, client);
        try {
            country = getMap("/org/xbib/interlibrary/z/hbz/country.json");
            language = getMap("/org/xbib/interlibrary/z/hbz/language.json");
            monographcodes = getMap("/org/xbib/interlibrary/z/hbz/monographcodes.json");
            serialcodes = getMap("/org/xbib/interlibrary/z/hbz/serialcodes.json");
            statuscodes = getMap("/org/xbib/interlibrary/z/hbz/status.json");
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    protected List<String> getServices() {
        return Arrays.asList("HBZ", "hbz");
    }

    @Override
    protected String getIdentifierField() {
        return "001";
    }

    @Override
    protected List<String> getCreatorFields() {
        return creatorFields;
    }

    @Override
    protected List<String> getTitleFields() {
        return titleFields;
    }

    @Override
    protected String getCountryField() {
        return "036";
    }

    @Override
    protected Map<String, Object> getCountryMap() {
        return country;
    }

    @Override
    protected String getLanguageField() {
        return "037";
    }

    @Override
    protected Map<String, Object> getLanguageMap() {
        return language;
    }

    @Override
    protected Map<String, Object> getMonographCodes() {
        return monographcodes;
    }

    @Override
    protected Map<String, Object> getSerialCodes() {
        return serialcodes;
    }

    @Override
    protected Map<String, Object> getStatusCodes() {
        return statuscodes;
    }

    @SuppressWarnings("unchecked")
    @Override
    protected void processAvailServices(AvailRequest request, InputStream inputStream,
                                        MultiMap<String, Map<String, Object>> multiMap,
                                        AvailResponse response) {
        String format = client.getSettings().get("connection.format");
        String type = client.getSettings().get("connection.type");
        Charset charset = Charset.forName(client.getSettings().get("connection.encoding"));
        Marc.Builder builder = Marc.builder()
                .setFormat(format)
                .setType(type)
                .setInputStream(inputStream)
                .setCharset(charset);
        Iterator<MarcRecord> marcRecordIterator = builder.recordIterator();
        while (marcRecordIterator.hasNext()) {
            MarcRecord marcRecord = marcRecordIterator.next();
            if (request.isBibliographicDescriptionEnabled()) {
                response.getBibliographicDescription().clear();
                response.getBibliographicDescription().putAll(marcRecord);
            }
            logger.log(Level.FINE, () -> "avail: record = " + marcRecord);
            String sourceid = marcRecord.getFields("001").get(0).getValue();
            String carrierType = isOnline(marcRecord) ? "online resource" : "volume";
            for (MarcField marcField : marcRecord.getFields("088")) {
                Map<String, Object> info = new LinkedHashMap<>();
                String sigel = marcField.getFirstSubfieldValue("a");
                StringBuilder sb = new StringBuilder();
                List<Map<String, Object>> list = new ArrayList<>();
                Map<String, Object> location = new HashMap<>();
                // we may have subfield repetitions, who knows?
                Map<String, Object> service = new LinkedHashMap<>();
                for (MarcField.Subfield subfield : marcField.getSubfields()) {
                    location = makeLocation(subfield, location, sb, list);
                }
                if (!location.isEmpty()) {
                    list.add(location);
                }
                info.put("location", list);
                service.put("info", info);
                String isil = interlibraryConfiguration.normalizeLibraryIdentifier(sigel);
                if (isil != null) {
                    service.put("isil", isil);
                    Library library = interlibraryConfiguration.getLibraries().get(isil);
                    if (library != null) {
                        service.put("name", library.getName());
                        service.put("domain", library.getDomain().getName());
                        service.put("organization", library.getDomain().getOrganization());
                    }
                    service.put("carriertype", carrierType);
                    service.put("type", "interlibrary");
                    String status = marcField.getFirstSubfieldValue("e");
                    if (status != null) {
                        List<String> codes = (List<String>) getStatusCodes().get(status);
                        if (codes != null) {
                            service.put("mode", codes);
                            if (codes.contains("none")) {
                                service.put("type", "noninterlibrary");
                            }
                        }
                    }
                    service.put("source", name);
                    service.put("sourceid", sourceid);
                    service.put("_id", isil + "/" + sb.toString());
                    multiMap.put(isil, service);
                }
            }
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    protected List<Map<String, Object>> extractServices(MarcRecord marcRecord) {
        List<Map<String, Object>> list = new ArrayList<>();
        marcRecord.getFields("088").forEach(marcField -> {
            String sigel = marcField.getFirstSubfieldValue("a");
            if (sigel != null) {
                String isil = interlibraryConfiguration.normalizeLibraryIdentifier(sigel);
                if (isil != null) {
                    Library library = interlibraryConfiguration.getLibraries().get(isil);
                    if (library != null) {
                        Map<String, Object> map = new LinkedHashMap<>();
                        map.put("isil", library.getISIL());
                        map.put("name", library.getName());
                        map.put("domain", library.getDomainName());
                        map.put("organization", library.getDomain().getOrganization());
                        putIf(map, "collection", marcField.getFirstSubfieldValue("b"));
                        putIf(map, "callnumber", marcField.getFirstSubfieldValue("c"));
                        putIf(map, "description", marcField.getFirstSubfieldValue("d"));
                        String status = marcField.getFirstSubfieldValue("e");
                        if (status != null) {
                            putIf(map, "status", status);
                            putIf(map, "code", (List<String>) getStatusCodes().get(status));
                        } else {
                            putIf(map, "code", Arrays.asList("copy", "loan"));
                        }
                        list.add(map);
                    } else {
                        logger.log(Level.WARNING, () -> "no library configured for ISIL: " + isil);
                    }
                } else {
                    logger.log(Level.WARNING, () -> "undefined identifier: " + sigel);
                }
            }
        });
        return list;
    }

    @Override
    protected List<Map<String, Object>> extractLinks(MarcRecord marcRecord) {
        List<Map<String, Object>> list = new ArrayList<>();
        marcRecord.getFields("655").forEach(marcField -> {
            Map<String, Object> map = new LinkedHashMap<>();
            putIf(map, "uri", marcField.getFirstSubfieldValue("u"));
            putIf(map, "publicnote", marcField.getFirstSubfieldValue("x"));
            putIf(map, "nonpublicnote", marcField.getFirstSubfieldValue("y"));
            list.add(map);
        });
        return list;
    }

    @SuppressWarnings("unchecked")
    @Override
    protected List<String> extractFormatCodes(MarcRecord marcRecord) {
        List<String> list = new ArrayList<>();
        marcRecord.getFields("051").forEach(marcField -> {
            String code = marcField.getFirstSubfieldValue(" ");
            for (Map.Entry<String, Object> entry : getMonographCodes().entrySet()) {
                int i = Integer.parseInt(entry.getKey());
                Map<String, String> map = (Map<String, String>) entry.getValue();
                String s = map.get(code.substring(i, i + 1));
                if (s != null) {
                    list.add(s);
                }
            }
        });
        marcRecord.getFields("052").forEach(marcField -> {
            String code = marcField.getFirstSubfieldValue(" ");
            for (Map.Entry<String, Object> entry : getSerialCodes().entrySet()) {
                int i = Integer.parseInt(entry.getKey());
                Map<String, String> map = (Map<String, String>) entry.getValue();
                String s = map.get(code.substring(i, i + 1));
                if (s != null) {
                    list.add(s);
                }
            }
        });
        return list;
    }

    private Map<String, Object> makeLocation(MarcField.Subfield subfield, Map<String, Object> location, StringBuilder sb,
                                             List<Map<String, Object>> list) {
        Map<String, Object> result = location;
        switch (subfield.getId()) {
            case "a":
                // already evaluated
                break;
            case "b":
                result = nextElement(list, location, "collection", subfield.getValue());
                if (sb.length() > 0) {
                    sb.append("/");
                }
                sb.append(subfield.getValue());
                break;
            case "c":
                result = nextElement(list, location, "callnumber", subfield.getValue());
                if (sb.length() > 0) {
                    sb.append("/");
                }
                sb.append(subfield.getValue());
                break;
            case "d":
                result = nextElement(list, location, "description", subfield.getValue());
                break;
            case "e":
                result = nextElement(list, location, "publicnote", subfield.getValue());
                break;
            default:
                logger.log(Level.WARNING, () -> "extra information in holdings field: " + subfield);
                break;
        }
        return result;
    }

    @Override
    protected boolean isOnline(MarcRecord marcRecord) {
        return marcRecord.getFields("062").stream().anyMatch(marcField ->
                isStartOf("cr", marcField.getFirstSubfieldValue("b"))) ||
                marcRecord.getFields("652").stream().anyMatch(marcField ->
                        isStartOf("Online-Ressource", marcField.getFirstSubfieldValue("a")));
    }

    private static final List<String> creatorFields = Arrays.asList(
            "100", "104", "108", "112", "116",
            "120", "124", "128", "132", "136",
            "140", "144", "148", "152", "156",
            "160", "164", "168", "172", "176",
            "180", "184", "188", "192", "196",
            "200", "204", "208", "212", "216",
            "220", "224", "228", "232", "236",
            "240", "244", "248", "252", "256",
            "260", "264", "268", "272", "276",
            "280", "284", "288", "292", "296");

    private static final List<String> titleFields = Arrays.asList(
            "310", "331", "335", "359", "360"
    );

    @Override
    public String getName() {
        return "HBZ";
    }
}
