package org.xbib.interlibrary.z.swb;

import org.xbib.content.settings.Settings;
import org.xbib.interlibrary.action.avail.AvailRequest;
import org.xbib.interlibrary.action.avail.AvailResponse;
import org.xbib.interlibrary.api.Library;
import org.xbib.interlibrary.api.service.ServiceArguments;
import org.xbib.interlibrary.common.util.MultiMap;
import org.xbib.interlibrary.z.AbstractZAvailService;
import org.xbib.interlibrary.z.ZClientHelper;
import org.xbib.marc.Marc;
import org.xbib.marc.MarcField;
import org.xbib.marc.MarcRecord;
import org.xbib.marc.label.RecordLabel;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 */
public class AvailService extends AbstractZAvailService {

    private static final Logger logger = Logger.getLogger(AvailService.class.getName());

    public AvailService(Settings settings,
                        ServiceArguments arguments,
                        ZClientHelper client) {
        super(settings, arguments, client);
    }

    @Override
    protected List<String> getServices() {
        return Arrays.asList("SWB", "swb");
    }

    @Override
    protected void processAvailServices(AvailRequest request, InputStream inputStream,
                                        MultiMap<String, Map<String, Object>> multiMap,
                                        AvailResponse response) {
        String format = client.getSettings().get("connection.format");
        String type = client.getSettings().get("connection.type");
        Charset charset = Charset.forName(client.getSettings().get("connection.encoding"));
        Marc.Builder builder = Marc.builder()
                .setFormat(format)
                .setType(type)
                .setInputStream(inputStream)
                .setCharset(charset);
        for (MarcRecord marcRecord : builder.iterable()) {
            if (request.isBibliographicDescriptionEnabled()) {
                response.getBibliographicDescription().clear();
                response.getBibliographicDescription().putAll(marcRecord);
            }
            logger.log(Level.FINE, () -> "avail: record = " + marcRecord);
            String sourceid = marcRecord.getFields("001").get(0).getValue();
            String carrierType = isOnline(marcRecord) ? "online resource" : "volume";
            List<MarcRecord> locs = extractLok(marcRecord);
            for (MarcRecord loc : locs) {
                String isil = null;
                String callnumber = null;
                Map<String, Object> service = new LinkedHashMap<>();
                for (MarcField marcField : loc.getFields("852")) {
                    String s = marcField.getFirstSubfieldValue("a");
                    // There are inoffcial sigel/ISIL. Example:
                    // https://wiki.bsz-bw.de/lib/exe/fetch.php?media=v-team:daten:ilnvergabe-extern.pdf
                    // 100x = Universitätsbibliographie Hohenheim
                    // https://wiki.bsz-bw.de/lib/exe/fetch.php?media=v-team:daten:liste-nationallizenzen_www.pdf
                    // ZDB1NTA = ?
                    if ("LFER".equals(s)) {
                        // skip place-holder electronic resource ISIL
                        continue;
                    }
                    if (s != null && !s.isEmpty()) {
                        isil = s;
                    }
                    String t = marcField.getFirstSubfieldValue("c");
                    if (t != null && !t.isEmpty()) {
                        callnumber = t;
                    }
                }
                if (isil != null) {
                    service.put("isil", isil);
                    Library library = interlibraryConfiguration.getLibraries().get(isil);
                    if (library != null) {
                        service.put("name", library.getName());
                        service.put("domain", library.getDomain().getName());
                        service.put("organization", library.getDomain().getOrganization());
                    }
                    service.put("carriertype", carrierType);
                    service.put("type", "interlibrary");
                    service.put("callnumber", callnumber);
                    service.put("source", name);
                    service.put("sourceid", sourceid);
                    service.put("_id", isil);
                    multiMap.put(isil, service);
                }
            }
        }
        logger.log(Level.FINE, () -> "multi map = " + multiMap);
    }

    @Override
    protected String getIdentifierField() {
        return null;
    }

    @Override
    protected List<String> getCreatorFields() {
        return null;
    }

    @Override
    protected List<String> getTitleFields() {
        return null;
    }

    @Override
    protected String getCountryField() {
        return null;
    }

    @Override
    protected Map<String, Object> getCountryMap() {
        return null;
    }

    @Override
    protected String getLanguageField() {
        return null;
    }

    @Override
    protected Map<String, Object> getLanguageMap() {
        return null;
    }

    @Override
    protected Map<String, Object> getMonographCodes() {
        return null;
    }

    @Override
    protected Map<String, Object> getSerialCodes() {
        return null;
    }

    @Override
    protected Map<String, Object> getStatusCodes() {
        return null;
    }

    @Override
    protected boolean isOnline(MarcRecord marcRecord) {
        // check MARC 007 "carrier type" for "cr"
        for (MarcField marcField : marcRecord.getFields("007")) {
            if (isStartOf("cr", marcField.getValue())) {
                return true;
            }
        }
        return false;
    }

    private List<MarcRecord> extractLok(MarcRecord marcRecord) {
        List<MarcRecord> marcRecords = new ArrayList<>();
        Marc.Builder marcBuilder = null;
        for (MarcField marcField : marcRecord.getFields("LOK")) {
            String value = marcField.getFirstSubfieldValue("0");
            if (value.startsWith("000")) {
                if (marcBuilder != null) {
                    marcRecords.add(marcBuilder.buildRecord());
                }
                marcBuilder = Marc.builder();
                RecordLabel recordLabel = RecordLabel.builder()
                        .from(value.substring(3).toCharArray())
                        .setSubfieldIdentifierLength(2)
                        .build();
                marcBuilder.recordLabel(recordLabel);
            } else {
                if (marcBuilder != null) {
                    MarcField.Builder builder = MarcField.builder();
                    builder.tag(value.substring(0, 3))
                            .indicator(value.substring(3, 5));
                    if (!value.substring(5).isEmpty()) {
                        builder.value(marcBuilder.getRecordLabel(), value.substring(5));
                    }
                    if (!marcField.getSubfields().isEmpty()) {
                        for (MarcField.Subfield subfield : marcField.getSubfields()) {
                            if (!subfield.getId().equals("0")) {
                                builder.subfield(subfield);
                            }
                        }
                    }
                    MarcField mf = builder.build();
                    marcBuilder.addField(mf);
                }
            }
        }
        if (marcBuilder != null) {
            marcRecords.add(marcBuilder.buildRecord());
        }
        return marcRecords;
    }

    @Override
    public String getName() {
        return "SWB";
    }
}
