/*
 * Decompiled with CFR 0.152.
 */
package org.xbib.io.compress.bgzf;

import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.SyncFailedException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

public class BinaryCodec {
    private OutputStream outputStream;
    private String outputFileName;
    private InputStream inputStream;
    private String inputFileName;
    private boolean isWriting;
    private ByteBuffer byteBuffer;
    private final byte[] scratchBuffer = new byte[16];
    private static final ByteOrder LITTLE_ENDIAN = ByteOrder.LITTLE_ENDIAN;
    private static final byte[] NULL_BYTE = new byte[]{0};
    private static final long MAX_UBYTE = 255L;
    private static final long MAX_USHORT = 65535L;
    private static final long MAX_UINT = 0xFFFFFFFFL;
    private static final int MAX_BYTE_BUFFER = 8;

    public BinaryCodec(File file, boolean writing) throws FileNotFoundException {
        this();
        this.isWriting = writing;
        if (this.isWriting) {
            this.outputStream = new FileOutputStream(file);
            this.outputFileName = file.getName();
        } else {
            this.inputStream = new FileInputStream(file);
            this.inputFileName = file.getName();
        }
    }

    public BinaryCodec(String fileName, boolean writing) throws FileNotFoundException {
        this(new File(fileName), writing);
    }

    public BinaryCodec(OutputStream outputStream) {
        this();
        this.setOutputStream(outputStream);
    }

    public BinaryCodec(InputStream inputStream) {
        this();
        this.setInputStream(inputStream);
    }

    public BinaryCodec() {
        this.initByteBuffer();
    }

    private void initByteBuffer() {
        this.byteBuffer = ByteBuffer.allocate(8);
        this.byteBuffer.order(LITTLE_ENDIAN);
    }

    private void writeByteBuffer(int numBytes) throws IOException {
        assert (numBytes <= this.byteBuffer.limit());
        this.writeBytes(this.byteBuffer.array(), 0, numBytes);
    }

    public void writeByte(byte bite) throws IOException {
        this.byteBuffer.clear();
        this.byteBuffer.put(bite);
        this.writeByteBuffer(1);
    }

    public void writeByte(int b) throws IOException {
        this.writeByte((byte)b);
    }

    public void writeBytes(byte[] bytes) throws IOException {
        this.writeBytes(bytes, 0, bytes.length);
    }

    public void writeBytes(byte[] bytes, int startOffset, int numBytes) throws IOException {
        if (!this.isWriting) {
            throw new IllegalStateException("Calling write method on BinaryCodec open for read.");
        }
        this.outputStream.write(bytes, startOffset, numBytes);
    }

    public void writeInt(int value) throws IOException {
        this.byteBuffer.clear();
        this.byteBuffer.putInt(value);
        this.writeByteBuffer(4);
    }

    public void writeDouble(double value) throws IOException {
        this.byteBuffer.clear();
        this.byteBuffer.putDouble(value);
        this.writeByteBuffer(8);
    }

    public void writeLong(long value) throws IOException {
        this.byteBuffer.clear();
        this.byteBuffer.putLong(value);
        this.writeByteBuffer(8);
    }

    public void writeShort(short value) throws IOException {
        this.byteBuffer.clear();
        this.byteBuffer.putShort(value);
        this.writeByteBuffer(2);
    }

    public void writeFloat(float value) throws IOException {
        this.byteBuffer.clear();
        this.byteBuffer.putFloat(value);
        this.writeByteBuffer(4);
    }

    public void writeBoolean(boolean value) throws IOException {
        this.byteBuffer.clear();
        this.byteBuffer.put(value ? (byte)1 : 0);
        this.writeByteBuffer(1);
    }

    public void writeString(String value, boolean writeLength, boolean appendNull) throws IOException {
        if (writeLength) {
            int lengthToWrite = value.length();
            if (appendNull) {
                ++lengthToWrite;
            }
            this.writeInt(lengthToWrite);
        }
        this.writeString(value);
        if (appendNull) {
            this.writeBytes(NULL_BYTE);
        }
    }

    private void writeString(String value) throws IOException {
        byte[] byteBuffer = new byte[value.length()];
        char[] charBuffer = value.toCharArray();
        for (int i = 0; i < charBuffer.length; ++i) {
            byteBuffer[i] = (byte)(charBuffer[i] & 0xFF);
        }
        this.writeBytes(byteBuffer);
    }

    public void writeUByte(short val) throws IOException {
        if (val < 0) {
            throw new IllegalArgumentException("Negative value (" + val + ") passed to unsigned writing method.");
        }
        if ((long)val > 255L) {
            throw new IllegalArgumentException("Value (" + val + ") to large to be written as ubyte.");
        }
        this.byteBuffer.clear();
        this.byteBuffer.putShort(val);
        this.writeByteBuffer(1);
    }

    public void writeUShort(int val) throws IOException {
        if (val < 0) {
            throw new IllegalArgumentException("Negative value (" + val + ") passed to unsigned writing method.");
        }
        if ((long)val > 65535L) {
            throw new IllegalArgumentException("Value (" + val + ") to large to be written as ushort.");
        }
        this.byteBuffer.clear();
        this.byteBuffer.putInt(val);
        this.writeByteBuffer(2);
    }

    public void writeUInt(long val) throws IOException {
        if (val < 0L) {
            throw new IllegalArgumentException("Negative value (" + val + ") passed to unsigned writing method.");
        }
        if (val > 0xFFFFFFFFL) {
            throw new IllegalArgumentException("Value (" + val + ") to large to be written as uint.");
        }
        this.byteBuffer.clear();
        this.byteBuffer.putLong(val);
        this.writeByteBuffer(4);
    }

    public void readBytes(byte[] buffer) throws IOException {
        this.readBytes(buffer, 0, buffer.length);
    }

    public void readBytes(byte[] buffer, int offset, int length) throws IOException {
        int numRead;
        int totalNumRead = 0;
        do {
            if ((numRead = this.readBytesOrFewer(buffer, offset + totalNumRead, length - totalNumRead)) >= 0) continue;
            throw new EOFException(this.constructErrorMessage("Premature EOF"));
        } while ((totalNumRead += numRead) < length);
    }

    public int readBytesOrFewer(byte[] buffer, int offset, int length) throws IOException {
        if (this.isWriting) {
            throw new IllegalStateException("Calling read method on BinaryCodec open for write.");
        }
        return this.inputStream.read(buffer, offset, length);
    }

    public byte readByte() throws IOException {
        if (this.isWriting) {
            throw new IllegalStateException("Calling read method on BinaryCodec open for write.");
        }
        int ret = this.inputStream.read();
        if (ret == -1) {
            throw new EOFException(this.constructErrorMessage("Premature EOF"));
        }
        return (byte)ret;
    }

    public boolean knownAtEof() throws IOException {
        if (this.isWriting) {
            throw new IllegalStateException("Calling knownAtEof method on BinaryCodec open for write.");
        }
        return this.inputStream instanceof ByteArrayInputStream && this.inputStream.available() == 0;
    }

    public String readString(int length) throws IOException {
        byte[] buffer = length <= this.scratchBuffer.length ? this.scratchBuffer : new byte[length];
        this.readBytes(buffer, 0, length);
        char[] charBuffer = new char[length];
        for (int i = 0; i < length; ++i) {
            charBuffer[i] = (char)buffer[i];
        }
        return new String(charBuffer);
    }

    public String readNullTerminatedString() throws IOException {
        StringBuilder ret = new StringBuilder();
        byte b = this.readByte();
        while (b != 0) {
            ret.append((char)(b & 0xFF));
            b = this.readByte();
        }
        return ret.toString();
    }

    public String readLengthAndString(boolean devourNull) throws IOException {
        int length = this.readInt();
        if (devourNull) {
            --length;
        }
        String ret = this.readString(length);
        if (devourNull) {
            this.readByte();
        }
        return ret;
    }

    private void readByteBuffer(int numBytes) throws IOException {
        assert (numBytes <= this.byteBuffer.capacity());
        this.readBytes(this.byteBuffer.array(), 0, numBytes);
        this.byteBuffer.limit(this.byteBuffer.capacity());
        this.byteBuffer.position(numBytes);
    }

    public int readInt() throws IOException {
        this.readByteBuffer(4);
        this.byteBuffer.flip();
        return this.byteBuffer.getInt();
    }

    public double readDouble() throws IOException {
        this.readByteBuffer(8);
        this.byteBuffer.flip();
        return this.byteBuffer.getDouble();
    }

    public long readLong() throws IOException {
        this.readByteBuffer(8);
        this.byteBuffer.flip();
        return this.byteBuffer.getLong();
    }

    public short readShort() throws IOException {
        this.readByteBuffer(2);
        this.byteBuffer.flip();
        return this.byteBuffer.getShort();
    }

    public float readFloat() throws IOException {
        this.readByteBuffer(4);
        this.byteBuffer.flip();
        return this.byteBuffer.getFloat();
    }

    public boolean readBoolean() throws IOException {
        return this.readByte() == 1;
    }

    public short readUByte() throws IOException {
        this.readByteBuffer(1);
        this.byteBuffer.put((byte)0);
        this.byteBuffer.flip();
        return this.byteBuffer.getShort();
    }

    public int readUShort() throws IOException {
        this.readByteBuffer(2);
        this.byteBuffer.putShort((short)0);
        this.byteBuffer.flip();
        return this.byteBuffer.getInt();
    }

    public long readUInt() throws IOException {
        this.readByteBuffer(4);
        this.byteBuffer.putInt(0);
        this.byteBuffer.flip();
        return this.byteBuffer.getLong();
    }

    public void close() throws IOException {
        if (this.isWriting) {
            if (this.outputStream instanceof FileOutputStream) {
                this.outputStream.flush();
                FileOutputStream fos = (FileOutputStream)this.outputStream;
                try {
                    fos.getFD().sync();
                }
                catch (SyncFailedException syncFailedException) {
                    // empty catch block
                }
            }
            this.outputStream.close();
        } else {
            this.inputStream.close();
        }
    }

    private String constructErrorMessage(String msg) {
        String filename;
        StringBuilder sb = new StringBuilder(msg);
        sb.append("; BinaryCodec in ");
        sb.append(this.isWriting ? "write" : "read");
        sb.append("mode; ");
        String string = filename = this.isWriting ? this.outputFileName : this.inputFileName;
        if (filename != null) {
            sb.append("file: ");
            sb.append(filename);
        } else {
            sb.append("streamed file (filename not available)");
        }
        return sb.toString();
    }

    public String getInputFileName() {
        return this.inputFileName;
    }

    public String getOutputFileName() {
        return this.outputFileName;
    }

    public void setOutputFileName(String outputFileName) {
        this.outputFileName = outputFileName;
    }

    public void setInputFileName(String inputFileName) {
        this.inputFileName = inputFileName;
    }

    public boolean isWriting() {
        return this.isWriting;
    }

    public OutputStream getOutputStream() {
        return this.outputStream;
    }

    public InputStream getInputStream() {
        return this.inputStream;
    }

    public void setInputStream(InputStream is) {
        this.isWriting = false;
        this.inputStream = is;
    }

    public void setOutputStream(OutputStream os) {
        this.isWriting = true;
        this.outputStream = os;
    }
}

