package org.xblackcat.sjpu.util.function;

import java.util.Comparator;
import java.util.Objects;

/**
 * Represents an operation upon two operands of the same type, producing a result
 * of the same type as the operands.  This is a specialization of
 * {@link BiFunctionEx} for the case where the operands and the result are all of
 * the same type.
 * <p>
 * <p>This is a <a href="package-summary.html">functional interface</a>
 * whose functional method is {@link #apply(Object, Object)}.
 *
 * @param <T> the type of the operands and result of the operator
 * @param <E> the type of exception could be thrown while performing operation
 * @see BiFunctionEx
 * @see UnaryOperatorEx
 * @since 1.8
 */
@FunctionalInterface
public interface BinaryOperatorEx<T, E extends Throwable> extends BiFunctionEx<T, T, T, E> {
    /**
     * Returns a {@link BinaryOperatorEx} which returns the lesser of two elements
     * according to the specified {@code Comparator}.
     *
     * @param <T>        the type of the input arguments of the comparator
     * @param comparator a {@code Comparator} for comparing the two values
     * @return a {@code BinaryOperator} which returns the lesser of its operands,
     * according to the supplied {@code Comparator}
     * @throws NullPointerException if the argument is null
     */
    static <T, E extends Throwable> BinaryOperatorEx<T, E> minBy(Comparator<? super T> comparator) {
        Objects.requireNonNull(comparator);
        return (a, b) -> comparator.compare(a, b) <= 0 ? a : b;
    }

    /**
     * Returns a {@link BinaryOperatorEx} which returns the greater of two elements
     * according to the specified {@code Comparator}.
     *
     * @param <T>        the type of the input arguments of the comparator
     * @param comparator a {@code Comparator} for comparing the two values
     * @return a {@code BinaryOperator} which returns the greater of its operands,
     * according to the supplied {@code Comparator}
     * @throws NullPointerException if the argument is null
     */
    static <T, E extends Throwable> BinaryOperatorEx<T, E> maxBy(Comparator<? super T> comparator) {
        Objects.requireNonNull(comparator);
        return (a, b) -> comparator.compare(a, b) >= 0 ? a : b;
    }

    default UnaryOperatorEx<T, E> fixRight(T u) {
        return t -> apply(t, u);
    }

    default UnaryOperatorEx<T, E> fixLeft(T t) {
        return u -> apply(t, u);
    }
}
