/*
 * Decompiled with CFR 0.152.
 */
package org.xipki.security.pkcs11;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECPoint;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.RuntimeCryptoException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xipki.pkcs11.wrapper.PKCS11Constants;
import org.xipki.pkcs11.wrapper.TokenException;
import org.xipki.security.HashAlgo;
import org.xipki.security.SignAlgo;
import org.xipki.security.XiContentSigner;
import org.xipki.security.XiSecurityException;
import org.xipki.security.pkcs11.DigestOutputStream;
import org.xipki.security.pkcs11.P11Key;
import org.xipki.security.pkcs11.P11Params;
import org.xipki.security.util.GMUtil;
import org.xipki.security.util.PKCS1Util;
import org.xipki.security.util.SignerUtil;
import org.xipki.util.Args;
import org.xipki.util.LogUtil;

abstract class P11ContentSigner
implements XiContentSigner {
    private static final Logger LOG = LoggerFactory.getLogger(P11ContentSigner.class);
    protected final P11Key identity;
    protected final SignAlgo signAlgo;
    protected final byte[] encodedAlgorithmIdentifier;

    private P11ContentSigner(P11Key identity, SignAlgo signAlgo) throws XiSecurityException {
        this.identity = (P11Key)Args.notNull((Object)identity, (String)"identity");
        this.signAlgo = (SignAlgo)((Object)Args.notNull((Object)((Object)signAlgo), (String)"signAlgo"));
        try {
            this.encodedAlgorithmIdentifier = signAlgo.getAlgorithmIdentifier().getEncoded();
        }
        catch (IOException ex) {
            throw new XiSecurityException("could not encode AlgorithmIdentifier", ex);
        }
    }

    static P11ContentSigner newInstance(P11Key identity, SignAlgo signAlgo, SecureRandom random, PublicKey publicKey) throws XiSecurityException {
        long keyType = identity.getKeyType();
        if (keyType == 0L) {
            if (signAlgo.isRSAPSSSigAlgo()) {
                return new RSAPSS(identity, signAlgo, random);
            }
            return new RSAPkcs1v1_5(identity, signAlgo);
        }
        if (keyType == 3L || keyType == 0xFFFFF001L) {
            boolean isSm2p256v1;
            boolean bl = isSm2p256v1 = keyType == 0xFFFFF001L || GMObjectIdentifiers.sm2p256v1.equals((ASN1Primitive)identity.getEcParams());
            if (isSm2p256v1) {
                if (publicKey == null) {
                    publicKey = Optional.ofNullable(identity.getPublicKey()).orElseThrow(() -> new XiSecurityException("SM2 signer needs public key, but could not get anyone."));
                }
                ECPoint w = ((ECPublicKey)publicKey).getW();
                BigInteger wx = w.getAffineX();
                BigInteger wy = w.getAffineY();
                return new SM2(identity, signAlgo, GMObjectIdentifiers.sm2p256v1, wx, wy);
            }
            return new ECDSA(identity, signAlgo);
        }
        if (keyType == 1L) {
            return new DSA(identity, signAlgo);
        }
        if (keyType == 64L) {
            return new EdDSA(identity, signAlgo);
        }
        if (keyType == 31L || keyType == 16L) {
            return new Mac(identity, signAlgo);
        }
        throw new XiSecurityException("unsupported key type " + PKCS11Constants.ckkCodeToName((long)keyType));
    }

    public final AlgorithmIdentifier getAlgorithmIdentifier() {
        return this.signAlgo.getAlgorithmIdentifier();
    }

    @Override
    public final byte[] getEncodedAlgorithmIdentifier() {
        return Arrays.copyOf(this.encodedAlgorithmIdentifier, this.encodedAlgorithmIdentifier.length);
    }

    private static class SM2
    extends P11ContentSigner {
        private static final Map<SignAlgo, Long> algoMechMap = new HashMap<SignAlgo, Long>();
        private final long mechanism;
        private final OutputStream outputStream;
        private final byte[] z;

        SM2(P11Key identity, SignAlgo signAlgo, ASN1ObjectIdentifier curveOid, BigInteger pubPointX, BigInteger pubPointY) throws XiSecurityException {
            super(identity, signAlgo);
            if (!signAlgo.isSM2SigAlgo()) {
                throw new XiSecurityException("not an SM2 algorithm: " + signAlgo);
            }
            HashAlgo hashAlgo = signAlgo.getHashAlgo();
            long mech = Optional.ofNullable(algoMechMap.get((Object)signAlgo)).orElseThrow(() -> new XiSecurityException("unsupported signature algorithm " + signAlgo));
            if (identity.supportsSign(0xFFFFF002L)) {
                this.mechanism = 0xFFFFF002L;
                this.z = GMUtil.getSM2Z(curveOid, pubPointX, pubPointY);
                this.outputStream = new DigestOutputStream((Digest)hashAlgo.createDigest());
            } else if (identity.supportsSign(mech)) {
                this.mechanism = mech;
                this.z = null;
                this.outputStream = new ByteArrayOutputStream();
            } else {
                throw new XiSecurityException("signature algorithm " + signAlgo + " is not supported by the device");
            }
        }

        public OutputStream getOutputStream() {
            this.reset();
            return this.outputStream;
        }

        private void reset() {
            if (this.outputStream instanceof ByteArrayOutputStream) {
                ((ByteArrayOutputStream)this.outputStream).reset();
            } else {
                ((DigestOutputStream)this.outputStream).reset();
                try {
                    this.outputStream.write(this.z, 0, this.z.length);
                }
                catch (IOException ex) {
                    throw new IllegalStateException(ex.getMessage());
                }
            }
        }

        public byte[] getSignature() {
            try {
                return SignerUtil.dsaSigPlainToX962(this.getPlainSignature());
            }
            catch (XiSecurityException ex) {
                LogUtil.warn((Logger)LOG, (Throwable)ex);
                throw new RuntimeCryptoException("XiSecurityException: " + ex.getMessage());
            }
            catch (Throwable th) {
                LogUtil.warn((Logger)LOG, (Throwable)th);
                throw new RuntimeCryptoException(th.getClass().getName() + ": " + th.getMessage());
            }
        }

        private byte[] getPlainSignature() throws TokenException {
            byte[] dataToSign;
            P11Params.P11ByteArrayParams params;
            if (this.outputStream instanceof ByteArrayOutputStream) {
                params = new P11Params.P11ByteArrayParams(GMUtil.getDefaultIDA());
                dataToSign = ((ByteArrayOutputStream)this.outputStream).toByteArray();
            } else {
                params = null;
                dataToSign = ((DigestOutputStream)this.outputStream).digest();
            }
            this.reset();
            return this.identity.sign(this.mechanism, params, dataToSign);
        }

        static {
            algoMechMap.put(SignAlgo.SM2_SM3, 0xFFFFF003L);
        }
    }

    private static class RSAPSS
    extends P11ContentSigner {
        private static final Map<SignAlgo, Long> algoMechMap = new HashMap<SignAlgo, Long>();
        private final long mechanism;
        private final P11Params.P11RSAPkcsPssParams parameters;
        private final OutputStream outputStream;
        private final SecureRandom random;

        RSAPSS(P11Key identity, SignAlgo signAlgo, SecureRandom random) throws XiSecurityException {
            super(identity, signAlgo);
            boolean usePss;
            if (!signAlgo.isRSAPSSSigAlgo()) {
                throw new XiSecurityException("not an RSA PSS algorithm: " + signAlgo);
            }
            this.random = (SecureRandom)Args.notNull((Object)random, (String)"random");
            HashAlgo hashAlgo = signAlgo.getHashAlgo();
            long mech = Optional.ofNullable(algoMechMap.get((Object)signAlgo)).orElseThrow(() -> new XiSecurityException("unsupported signature algorithm " + signAlgo));
            boolean bl = usePss = signAlgo.isRSAPSSMGF1SigAlgo() && identity.supportsSign(13L);
            if (usePss) {
                switch (signAlgo.getHashAlgo()) {
                    case SHA1: 
                    case SHA224: 
                    case SHA256: 
                    case SHA384: 
                    case SHA512: {
                        break;
                    }
                    default: {
                        usePss = false;
                    }
                }
            }
            if (usePss) {
                this.mechanism = 13L;
                this.parameters = new P11Params.P11RSAPkcsPssParams(hashAlgo);
                this.outputStream = new DigestOutputStream((Digest)hashAlgo.createDigest());
            } else if (identity.supportsSign(3L)) {
                this.mechanism = 3L;
                this.parameters = null;
                this.outputStream = new DigestOutputStream((Digest)hashAlgo.createDigest());
            } else if (identity.supportsSign(mech)) {
                this.mechanism = mech;
                this.parameters = new P11Params.P11RSAPkcsPssParams(hashAlgo);
                this.outputStream = new ByteArrayOutputStream();
            } else {
                throw new XiSecurityException("signature algorithm " + signAlgo + " is not supported by the device");
            }
        }

        public OutputStream getOutputStream() {
            if (this.outputStream instanceof ByteArrayOutputStream) {
                ((ByteArrayOutputStream)this.outputStream).reset();
            } else {
                ((DigestOutputStream)this.outputStream).reset();
            }
            return this.outputStream;
        }

        public byte[] getSignature() {
            try {
                if (this.mechanism == 3L) {
                    byte[] encodedHashValue;
                    HashAlgo hash = this.signAlgo.getHashAlgo();
                    byte[] hashValue = ((DigestOutputStream)this.outputStream).digest();
                    try {
                        encodedHashValue = PKCS1Util.EMSA_PSS_ENCODE(hash, hashValue, hash, hash.getLength(), this.identity.getRsaModulus().bitLength(), this.random);
                    }
                    catch (XiSecurityException ex) {
                        throw new TokenException("XiSecurityException: " + ex.getMessage(), (Exception)ex);
                    }
                    return this.identity.sign(this.mechanism, this.parameters, encodedHashValue);
                }
                byte[] dataToSign = this.outputStream instanceof ByteArrayOutputStream ? ((ByteArrayOutputStream)this.outputStream).toByteArray() : ((DigestOutputStream)this.outputStream).digest();
                return this.identity.sign(this.mechanism, this.parameters, dataToSign);
            }
            catch (TokenException ex) {
                LogUtil.warn((Logger)LOG, (Throwable)ex, (String)"could not sign");
                throw new RuntimeCryptoException("SignerException: " + ex.getMessage());
            }
        }

        static {
            algoMechMap.put(SignAlgo.RSAPSS_SHA1, 14L);
            algoMechMap.put(SignAlgo.RSAPSS_SHA224, 71L);
            algoMechMap.put(SignAlgo.RSAPSS_SHA256, 67L);
            algoMechMap.put(SignAlgo.RSAPSS_SHA384, 68L);
            algoMechMap.put(SignAlgo.RSAPSS_SHA512, 69L);
            algoMechMap.put(SignAlgo.RSAPSS_SHA3_224, 103L);
            algoMechMap.put(SignAlgo.RSAPSS_SHA3_256, 99L);
            algoMechMap.put(SignAlgo.RSAPSS_SHA3_384, 100L);
            algoMechMap.put(SignAlgo.RSAPSS_SHA3_512, 101L);
        }
    }

    private static class RSAPkcs1v1_5
    extends P11ContentSigner {
        private static final Map<SignAlgo, Long> algoMechMap = new HashMap<SignAlgo, Long>();
        private final long mechanism;
        private final OutputStream outputStream;
        private final byte[] digestPkcsPrefix;
        private final int modulusBitLen;

        RSAPkcs1v1_5(P11Key identity, SignAlgo signAlgo) throws XiSecurityException {
            super(identity, signAlgo);
            if (!signAlgo.isRSAPkcs1SigAlgo()) {
                throw new XiSecurityException("not an RSA PKCS#1 algorithm: " + signAlgo);
            }
            long mech = Optional.ofNullable(algoMechMap.get((Object)signAlgo)).orElseThrow(() -> new XiSecurityException("unsupported signature algorithm " + signAlgo));
            if (identity.supportsSign(1L)) {
                this.mechanism = 1L;
            } else if (identity.supportsSign(3L)) {
                this.mechanism = 3L;
            } else if (identity.supportsSign(mech)) {
                this.mechanism = mech;
            } else {
                throw new XiSecurityException("signature algorithm " + signAlgo + " is not supported by the device");
            }
            if (this.mechanism == 1L || this.mechanism == 3L) {
                this.digestPkcsPrefix = PKCS1Util.getDigestPkcsPrefix(signAlgo.getHashAlgo());
                this.outputStream = new DigestOutputStream((Digest)signAlgo.getHashAlgo().createDigest());
            } else {
                this.digestPkcsPrefix = null;
                this.outputStream = new ByteArrayOutputStream();
            }
            this.modulusBitLen = identity.getRsaModulus().bitLength();
        }

        public OutputStream getOutputStream() {
            if (this.outputStream instanceof ByteArrayOutputStream) {
                ((ByteArrayOutputStream)this.outputStream).reset();
            } else {
                ((DigestOutputStream)this.outputStream).reset();
            }
            return this.outputStream;
        }

        public byte[] getSignature() {
            byte[] dataToSign;
            if (this.outputStream instanceof ByteArrayOutputStream) {
                dataToSign = ((ByteArrayOutputStream)this.outputStream).toByteArray();
                ((ByteArrayOutputStream)this.outputStream).reset();
            } else {
                byte[] hashValue = ((DigestOutputStream)this.outputStream).digest();
                ((DigestOutputStream)this.outputStream).reset();
                dataToSign = new byte[this.digestPkcsPrefix.length + hashValue.length];
                System.arraycopy(this.digestPkcsPrefix, 0, dataToSign, 0, this.digestPkcsPrefix.length);
                System.arraycopy(hashValue, 0, dataToSign, this.digestPkcsPrefix.length, hashValue.length);
            }
            try {
                if (this.mechanism == 3L) {
                    dataToSign = PKCS1Util.EMSA_PKCS1_v1_5_encoding(dataToSign, this.modulusBitLen);
                }
                return this.identity.sign(this.mechanism, null, dataToSign);
            }
            catch (TokenException | XiSecurityException ex) {
                LogUtil.error((Logger)LOG, (Throwable)ex, (String)"could not sign");
                throw new RuntimeCryptoException("SignerException: " + ex.getMessage());
            }
        }

        static {
            algoMechMap.put(SignAlgo.RSA_SHA1, 6L);
            algoMechMap.put(SignAlgo.RSA_SHA224, 70L);
            algoMechMap.put(SignAlgo.RSA_SHA256, 64L);
            algoMechMap.put(SignAlgo.RSA_SHA384, 65L);
            algoMechMap.put(SignAlgo.RSA_SHA512, 66L);
            algoMechMap.put(SignAlgo.RSA_SHA3_224, 102L);
            algoMechMap.put(SignAlgo.RSA_SHA3_256, 96L);
            algoMechMap.put(SignAlgo.RSA_SHA3_384, 97L);
            algoMechMap.put(SignAlgo.RSA_SHA3_512, 98L);
        }
    }

    private static class Mac
    extends P11ContentSigner {
        private static final Map<SignAlgo, Long> algoMechMap = new HashMap<SignAlgo, Long>();
        private final long mechanism;
        private final ByteArrayOutputStream outputStream;

        Mac(P11Key identity, SignAlgo signAlgo) throws XiSecurityException {
            super(identity, signAlgo);
            long mech;
            this.mechanism = mech = Optional.ofNullable(algoMechMap.get((Object)signAlgo)).orElseThrow(() -> new XiSecurityException("Unsupported signature algorithm " + signAlgo)).longValue();
            if (identity.supportsSign(this.mechanism)) {
                throw new XiSecurityException("unsupported MAC algorithm " + signAlgo);
            }
            this.outputStream = new ByteArrayOutputStream();
        }

        public OutputStream getOutputStream() {
            this.outputStream.reset();
            return this.outputStream;
        }

        public byte[] getSignature() {
            try {
                byte[] dataToSign = this.outputStream.toByteArray();
                this.outputStream.reset();
                return this.identity.sign(this.mechanism, null, dataToSign);
            }
            catch (TokenException ex) {
                LogUtil.warn((Logger)LOG, (Throwable)ex);
                throw new RuntimeCryptoException("TokenException: " + ex.getMessage());
            }
            catch (Throwable th) {
                LogUtil.warn((Logger)LOG, (Throwable)th);
                throw new RuntimeCryptoException(th.getClass().getName() + ": " + th.getMessage());
            }
        }

        static {
            algoMechMap.put(SignAlgo.HMAC_SHA1, 545L);
            algoMechMap.put(SignAlgo.HMAC_SHA224, 598L);
            algoMechMap.put(SignAlgo.HMAC_SHA256, 593L);
            algoMechMap.put(SignAlgo.HMAC_SHA384, 609L);
            algoMechMap.put(SignAlgo.HMAC_SHA512, 625L);
            algoMechMap.put(SignAlgo.HMAC_SHA3_224, 694L);
            algoMechMap.put(SignAlgo.HMAC_SHA3_256, 689L);
            algoMechMap.put(SignAlgo.HMAC_SHA3_384, 705L);
            algoMechMap.put(SignAlgo.HMAC_SHA3_512, 721L);
        }
    }

    private static class EdDSA
    extends P11ContentSigner {
        private final ByteArrayOutputStream outputStream;
        private final long mechanism;

        EdDSA(P11Key identity, SignAlgo signAlgo) throws XiSecurityException {
            super(identity, signAlgo);
            if (SignAlgo.ED25519 != signAlgo) {
                throw new XiSecurityException("unsupported signature algorithm " + signAlgo);
            }
            this.mechanism = 4183L;
            if (!identity.supportsSign(this.mechanism)) {
                throw new XiSecurityException("unsupported signature algorithm " + signAlgo);
            }
            this.outputStream = new ByteArrayOutputStream();
        }

        public OutputStream getOutputStream() {
            this.outputStream.reset();
            return this.outputStream;
        }

        public byte[] getSignature() {
            byte[] content = this.outputStream.toByteArray();
            this.outputStream.reset();
            try {
                return this.identity.sign(this.mechanism, null, content);
            }
            catch (Throwable th) {
                LogUtil.warn((Logger)LOG, (Throwable)th);
                throw new RuntimeCryptoException(th.getClass().getName() + ": " + th.getMessage());
            }
        }
    }

    private static class ECDSA
    extends P11ContentSigner {
        private static final Map<SignAlgo, Long> algoMechMap = new HashMap<SignAlgo, Long>();
        private final OutputStream outputStream;
        private final long mechanism;

        ECDSA(P11Key identity, SignAlgo signAlgo) throws XiSecurityException {
            super(identity, signAlgo);
            if (!signAlgo.isECDSASigAlgo()) {
                throw new XiSecurityException("not an ECDSA algorithm: " + signAlgo);
            }
            long mech = Optional.ofNullable(algoMechMap.get((Object)signAlgo)).orElseThrow(() -> new XiSecurityException("unsupported signature algorithm " + signAlgo));
            if (identity.supportsSign(4161L)) {
                this.mechanism = 4161L;
                this.outputStream = new DigestOutputStream((Digest)signAlgo.getHashAlgo().createDigest());
            } else if (identity.supportsSign(mech)) {
                this.mechanism = mech;
                this.outputStream = new ByteArrayOutputStream();
            } else {
                throw new XiSecurityException("signature algorithm " + signAlgo + " is not supported by the device");
            }
        }

        public OutputStream getOutputStream() {
            if (this.outputStream instanceof ByteArrayOutputStream) {
                ((ByteArrayOutputStream)this.outputStream).reset();
            } else {
                ((DigestOutputStream)this.outputStream).reset();
            }
            return this.outputStream;
        }

        public byte[] getSignature() {
            try {
                byte[] plainSignature = this.getPlainSignature();
                return this.signAlgo.isPlainECDSASigAlgo() ? plainSignature : SignerUtil.dsaSigPlainToX962(plainSignature);
            }
            catch (XiSecurityException ex) {
                LogUtil.warn((Logger)LOG, (Throwable)ex);
                throw new RuntimeCryptoException("XiSecurityException: " + ex.getMessage());
            }
            catch (Throwable th) {
                LogUtil.warn((Logger)LOG, (Throwable)th);
                throw new RuntimeCryptoException(th.getClass().getName() + ": " + th.getMessage());
            }
        }

        private byte[] getPlainSignature() throws TokenException {
            byte[] dataToSign;
            if (this.outputStream instanceof ByteArrayOutputStream) {
                dataToSign = ((ByteArrayOutputStream)this.outputStream).toByteArray();
                ((ByteArrayOutputStream)this.outputStream).reset();
            } else {
                dataToSign = ((DigestOutputStream)this.outputStream).digest();
                ((DigestOutputStream)this.outputStream).reset();
            }
            return this.identity.sign(this.mechanism, null, dataToSign);
        }

        static {
            algoMechMap.put(SignAlgo.ECDSA_SHA1, 4162L);
            algoMechMap.put(SignAlgo.ECDSA_SHA224, 4163L);
            algoMechMap.put(SignAlgo.ECDSA_SHA256, 4164L);
            algoMechMap.put(SignAlgo.ECDSA_SHA384, 4165L);
            algoMechMap.put(SignAlgo.ECDSA_SHA512, 4166L);
            algoMechMap.put(SignAlgo.ECDSA_SHA3_224, 4167L);
            algoMechMap.put(SignAlgo.ECDSA_SHA3_256, 4168L);
            algoMechMap.put(SignAlgo.ECDSA_SHA3_384, 4169L);
            algoMechMap.put(SignAlgo.ECDSA_SHA3_512, 4170L);
        }
    }

    private static class DSA
    extends P11ContentSigner {
        private static final Map<SignAlgo, Long> algoMechMap = new HashMap<SignAlgo, Long>();
        private final OutputStream outputStream;
        private final long mechanism;

        DSA(P11Key identity, SignAlgo signAlgo) throws XiSecurityException {
            super(identity, signAlgo);
            if (!signAlgo.isDSASigAlgo()) {
                throw new XiSecurityException("not a DSA algorithm: " + signAlgo);
            }
            long mech = Optional.ofNullable(algoMechMap.get((Object)signAlgo)).orElseThrow(() -> new XiSecurityException("unsupported signature algorithm " + signAlgo));
            if (identity.supportsSign(17L)) {
                this.mechanism = 17L;
                this.outputStream = new DigestOutputStream((Digest)signAlgo.getHashAlgo().createDigest());
            } else if (identity.supportsSign(mech)) {
                this.mechanism = mech;
                this.outputStream = new ByteArrayOutputStream();
            } else {
                throw new XiSecurityException("signature algorithm " + signAlgo + " is not supported by the device");
            }
        }

        public OutputStream getOutputStream() {
            if (this.outputStream instanceof ByteArrayOutputStream) {
                ((ByteArrayOutputStream)this.outputStream).reset();
            } else {
                ((DigestOutputStream)this.outputStream).reset();
            }
            return this.outputStream;
        }

        public byte[] getSignature() {
            try {
                byte[] plainSignature = this.getPlainSignature();
                return SignerUtil.dsaSigPlainToX962(plainSignature);
            }
            catch (XiSecurityException ex) {
                LogUtil.warn((Logger)LOG, (Throwable)ex);
                throw new RuntimeCryptoException("XiSecurityException: " + ex.getMessage());
            }
            catch (Throwable th) {
                LogUtil.warn((Logger)LOG, (Throwable)th);
                throw new RuntimeCryptoException(th.getClass().getName() + ": " + th.getMessage());
            }
        }

        private byte[] getPlainSignature() throws TokenException {
            byte[] dataToSign;
            if (this.outputStream instanceof ByteArrayOutputStream) {
                dataToSign = ((ByteArrayOutputStream)this.outputStream).toByteArray();
                ((ByteArrayOutputStream)this.outputStream).reset();
            } else {
                dataToSign = ((DigestOutputStream)this.outputStream).digest();
                ((DigestOutputStream)this.outputStream).reset();
            }
            return this.identity.sign(this.mechanism, null, dataToSign);
        }

        static {
            algoMechMap.put(SignAlgo.DSA_SHA1, 18L);
            algoMechMap.put(SignAlgo.DSA_SHA224, 19L);
            algoMechMap.put(SignAlgo.DSA_SHA256, 20L);
            algoMechMap.put(SignAlgo.DSA_SHA384, 21L);
            algoMechMap.put(SignAlgo.DSA_SHA512, 22L);
            algoMechMap.put(SignAlgo.DSA_SHA3_224, 24L);
            algoMechMap.put(SignAlgo.DSA_SHA3_256, 25L);
            algoMechMap.put(SignAlgo.DSA_SHA3_384, 26L);
            algoMechMap.put(SignAlgo.DSA_SHA3_512, 27L);
        }
    }
}

