/*
 * Decompiled with CFR 0.152.
 */
package org.xipki.qa.ca;

import java.security.spec.InvalidKeySpecException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.jcajce.provider.asymmetric.util.ECUtil;
import org.bouncycastle.math.ec.ECCurve;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xipki.ca.api.BadCertTemplateException;
import org.xipki.ca.api.profile.CertprofileException;
import org.xipki.ca.api.profile.KeyParametersOption;
import org.xipki.qa.ValidationIssue;
import org.xipki.security.util.AlgorithmUtil;
import org.xipki.security.util.X509Util;
import org.xipki.util.Args;
import org.xipki.util.CollectionUtil;
import org.xipki.util.LruCache;

public class PublicKeyChecker {
    private static final Logger LOG = LoggerFactory.getLogger(PublicKeyChecker.class);
    private static final LruCache<ASN1ObjectIdentifier, Integer> EC_CURVEFIELD_SIZES = new LruCache(100);
    private Map<ASN1ObjectIdentifier, KeyParametersOption> keyAlgorithms;

    public PublicKeyChecker(Map<ASN1ObjectIdentifier, KeyParametersOption> keyAlgorithms) throws CertprofileException {
        this.keyAlgorithms = keyAlgorithms;
    }

    public List<ValidationIssue> checkPublicKey(SubjectPublicKeyInfo publicKey, SubjectPublicKeyInfo requestedPublicKey) {
        ValidationIssue issue;
        Args.notNull((Object)publicKey, (String)"publicKey");
        Args.notNull((Object)requestedPublicKey, (String)"requestedPublicKey");
        LinkedList<ValidationIssue> resultIssues = new LinkedList<ValidationIssue>();
        if (this.keyAlgorithms != null) {
            issue = new ValidationIssue("X509.PUBKEY.SYN", "whether the public key in certificate is permitted");
            resultIssues.add(issue);
            try {
                this.checkPublicKey(publicKey);
            }
            catch (BadCertTemplateException ex) {
                issue.setFailureMessage(ex.getMessage());
            }
        }
        issue = new ValidationIssue("X509.PUBKEY.REQ", "whether public key matches the request one");
        resultIssues.add(issue);
        try {
            SubjectPublicKeyInfo c14nRequestedPublicKey = X509Util.toRfc3279Style((SubjectPublicKeyInfo)requestedPublicKey);
            if (!c14nRequestedPublicKey.equals((Object)publicKey)) {
                issue.setFailureMessage("public key in the certificate does not equal the requested one");
            }
        }
        catch (InvalidKeySpecException ex) {
            issue.setFailureMessage("public key in request is invalid");
        }
        return resultIssues;
    }

    private void checkPublicKey(SubjectPublicKeyInfo publicKey) throws BadCertTemplateException {
        if (CollectionUtil.isEmpty(this.keyAlgorithms)) {
            return;
        }
        ASN1ObjectIdentifier keyType = publicKey.getAlgorithm().getAlgorithm();
        if (!this.keyAlgorithms.containsKey(keyType)) {
            throw new BadCertTemplateException("key type " + keyType.getId() + " is not permitted");
        }
        KeyParametersOption keyParamsOption = this.keyAlgorithms.get(keyType);
        if (keyParamsOption instanceof KeyParametersOption.AllowAllParametersOption) {
            return;
        }
        if (keyParamsOption instanceof KeyParametersOption.ECParamatersOption) {
            ASN1ObjectIdentifier curveOid;
            KeyParametersOption.ECParamatersOption ecOption = (KeyParametersOption.ECParamatersOption)keyParamsOption;
            ASN1Encodable algParam = publicKey.getAlgorithm().getParameters();
            if (algParam instanceof ASN1ObjectIdentifier) {
                curveOid = (ASN1ObjectIdentifier)algParam;
                if (!ecOption.allowsCurve(curveOid)) {
                    throw new BadCertTemplateException("EC curve " + AlgorithmUtil.getCurveName((ASN1ObjectIdentifier)curveOid) + " (OID: " + curveOid.getId() + ") is not allowed");
                }
            } else {
                throw new BadCertTemplateException("only namedCurve EC public key is supported");
            }
            if (ecOption.pointEncodings() != null) {
                byte[] keyData = publicKey.getPublicKeyData().getBytes();
                if (keyData.length < 1) {
                    throw new BadCertTemplateException("invalid publicKeyData");
                }
                byte pointEncoding = keyData[0];
                if (!ecOption.pointEncodings().contains(pointEncoding)) {
                    throw new BadCertTemplateException("not-accepted EC point encoding " + pointEncoding);
                }
            }
            try {
                PublicKeyChecker.checkECSubjectPublicKeyInfo(curveOid, publicKey.getPublicKeyData().getBytes());
            }
            catch (BadCertTemplateException ex) {
                throw ex;
            }
            catch (Exception ex) {
                LOG.debug("checkECSubjectPublicKeyInfo", (Throwable)ex);
                throw new BadCertTemplateException("invalid public key: " + ex.getMessage());
            }
            return;
        }
        if (keyParamsOption instanceof KeyParametersOption.RSAParametersOption) {
            ASN1Integer modulus;
            KeyParametersOption.RSAParametersOption rsaOption = (KeyParametersOption.RSAParametersOption)keyParamsOption;
            try {
                ASN1Sequence seq = ASN1Sequence.getInstance((Object)publicKey.getPublicKeyData().getBytes());
                modulus = ASN1Integer.getInstance((Object)seq.getObjectAt(0));
            }
            catch (IllegalArgumentException ex) {
                throw new BadCertTemplateException("invalid publicKeyData");
            }
            int modulusLength = modulus.getPositiveValue().bitLength();
            if (rsaOption.allowsModulusLength(modulusLength)) {
                return;
            }
        } else if (keyParamsOption instanceof KeyParametersOption.DSAParametersOption) {
            int qlength;
            int plength;
            KeyParametersOption.DSAParametersOption dsaOption = (KeyParametersOption.DSAParametersOption)keyParamsOption;
            ASN1Encodable params = publicKey.getAlgorithm().getParameters();
            if (params == null) {
                throw new BadCertTemplateException("null Dss-Parms is not permitted");
            }
            try {
                ASN1Sequence seq = ASN1Sequence.getInstance((Object)params);
                ASN1Integer p = ASN1Integer.getInstance((Object)seq.getObjectAt(0));
                ASN1Integer q = ASN1Integer.getInstance((Object)seq.getObjectAt(1));
                plength = p.getPositiveValue().bitLength();
                qlength = q.getPositiveValue().bitLength();
            }
            catch (ArrayIndexOutOfBoundsException | IllegalArgumentException ex) {
                throw new BadCertTemplateException("illegal Dss-Parms");
            }
            boolean match = dsaOption.allowsPlength(plength);
            if (match) {
                match = dsaOption.allowsQlength(qlength);
            }
            if (match) {
                return;
            }
        } else {
            String txt = keyParamsOption == null ? "null" : keyParamsOption.getClass().getName();
            throw new IllegalStateException("should not reach here, unknown keyParamsOption " + txt);
        }
        throw new BadCertTemplateException("the given publicKey is not permitted");
    }

    private static void checkECSubjectPublicKeyInfo(ASN1ObjectIdentifier curveOid, byte[] encoded) throws BadCertTemplateException {
        Integer expectedLength = (Integer)EC_CURVEFIELD_SIZES.get((Object)curveOid);
        if (expectedLength == null) {
            X9ECParameters ecP = ECUtil.getNamedCurveByOid((ASN1ObjectIdentifier)curveOid);
            ECCurve curve = ecP.getCurve();
            expectedLength = (curve.getFieldSize() + 7) / 8;
            EC_CURVEFIELD_SIZES.put((Object)curveOid, (Object)expectedLength);
        }
        switch (encoded[0]) {
            case 2: 
            case 3: {
                if (encoded.length == expectedLength + 1) break;
                throw new BadCertTemplateException("incorrect length for compressed encoding");
            }
            case 4: 
            case 6: 
            case 7: {
                if (encoded.length == 2 * expectedLength + 1) break;
                throw new BadCertTemplateException("incorrect length for uncompressed/hybrid encoding");
            }
            default: {
                throw new BadCertTemplateException("invalid point encoding 0x" + Integer.toString(encoded[0], 16));
            }
        }
    }
}

