/*
 * Decompiled with CFR 0.152.
 */
package org.xipki.scep.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CRLException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Set;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DERPrintableString;
import org.bouncycastle.asn1.cms.Attribute;
import org.bouncycastle.asn1.cms.AttributeTable;
import org.bouncycastle.asn1.cms.SignedData;
import org.bouncycastle.asn1.nist.NISTObjectIdentifiers;
import org.bouncycastle.asn1.pkcs.CertificationRequest;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.pkcs.RSAPublicKey;
import org.bouncycastle.asn1.pkcs.RSASSAPSSparams;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.AuthorityKeyIdentifier;
import org.bouncycastle.asn1.x509.Certificate;
import org.bouncycastle.asn1.x509.CertificateList;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.Extensions;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.asn1.x509.X509ObjectIdentifiers;
import org.bouncycastle.cert.CertIOException;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaCertStore;
import org.bouncycastle.cms.CMSException;
import org.bouncycastle.cms.CMSSignedDataGenerator;
import org.bouncycastle.jce.X509KeyUsage;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.pkcs.PKCS10CertificationRequestBuilder;
import org.bouncycastle.util.Store;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xipki.scep.crypto.KeyUsage;
import org.xipki.scep.crypto.ScepHashAlgoType;

public class ScepUtil {
    private static final Logger LOG = LoggerFactory.getLogger(ScepUtil.class);
    private static final long MIN_IN_MS = 60000L;
    private static final long DAY_IN_MS = 86400000L;
    private static CertificateFactory certFact;
    private static Object certFactLock;

    private ScepUtil() {
    }

    public static SubjectPublicKeyInfo createSubjectPublicKeyInfo(PublicKey publicKey) throws IOException {
        ScepUtil.requireNonNull("publicKey", publicKey);
        if (publicKey instanceof java.security.interfaces.RSAPublicKey) {
            java.security.interfaces.RSAPublicKey rsaPubKey = (java.security.interfaces.RSAPublicKey)publicKey;
            return new SubjectPublicKeyInfo(new AlgorithmIdentifier(PKCSObjectIdentifiers.rsaEncryption, (ASN1Encodable)DERNull.INSTANCE), (ASN1Encodable)new RSAPublicKey(rsaPubKey.getModulus(), rsaPubKey.getPublicExponent()));
        }
        throw new IllegalArgumentException("unsupported public key " + publicKey);
    }

    public static PKCS10CertificationRequest generateRequest(PrivateKey privatekey, SubjectPublicKeyInfo subjectPublicKeyInfo, X500Name subjectDn, Map<ASN1ObjectIdentifier, ASN1Encodable> attributes) throws OperatorCreationException {
        ScepUtil.requireNonNull("privatekey", privatekey);
        ScepUtil.requireNonNull("subjectPublicKeyInfo", subjectPublicKeyInfo);
        ScepUtil.requireNonNull("subjectDn", subjectDn);
        PKCS10CertificationRequestBuilder csrBuilder = new PKCS10CertificationRequestBuilder(subjectDn, subjectPublicKeyInfo);
        if (attributes != null) {
            for (ASN1ObjectIdentifier attrType : attributes.keySet()) {
                csrBuilder.addAttribute(attrType, attributes.get(attrType));
            }
        }
        ContentSigner contentSigner = new JcaContentSignerBuilder(ScepUtil.getSignatureAlgorithm(privatekey, ScepHashAlgoType.SHA1)).build(privatekey);
        return csrBuilder.build(contentSigner);
    }

    public static PKCS10CertificationRequest generateRequest(PrivateKey privatekey, SubjectPublicKeyInfo subjectPublicKeyInfo, X500Name subjectDn, String challengePassword, List<Extension> extensions) throws OperatorCreationException {
        ScepUtil.requireNonNull("privatekey", privatekey);
        ScepUtil.requireNonNull("subjectPublicKeyInfo", subjectPublicKeyInfo);
        ScepUtil.requireNonNull("subjectDn", subjectDn);
        HashMap<ASN1ObjectIdentifier, ASN1Encodable> attributes = new HashMap<ASN1ObjectIdentifier, ASN1Encodable>();
        if (challengePassword != null && !challengePassword.isEmpty()) {
            DERPrintableString asn1Pwd = new DERPrintableString(challengePassword);
            attributes.put(PKCSObjectIdentifiers.pkcs_9_at_challengePassword, (ASN1Encodable)asn1Pwd);
        }
        if (extensions != null && !extensions.isEmpty()) {
            Extensions asn1Extensions = new Extensions(extensions.toArray(new Extension[0]));
            attributes.put(PKCSObjectIdentifiers.pkcs_9_at_extensionRequest, (ASN1Encodable)asn1Extensions);
        }
        return ScepUtil.generateRequest(privatekey, subjectPublicKeyInfo, subjectDn, attributes);
    }

    public static X509Certificate generateSelfsignedCert(CertificationRequest csr, PrivateKey identityKey) throws CertificateException {
        ScepUtil.requireNonNull("csr", csr);
        return ScepUtil.generateSelfsignedCert(csr.getCertificationRequestInfo().getSubject(), csr.getCertificationRequestInfo().getSubjectPublicKeyInfo(), identityKey);
    }

    public static X509Certificate generateSelfsignedCert(X500Name subjectDn, PublicKey pubKey, PrivateKey identityKey) throws CertificateException {
        SubjectPublicKeyInfo pubKeyInfo;
        try {
            pubKeyInfo = ScepUtil.createSubjectPublicKeyInfo(pubKey);
        }
        catch (IOException ex) {
            throw new CertificateException(ex.getMessage(), ex);
        }
        return ScepUtil.generateSelfsignedCert(subjectDn, pubKeyInfo, identityKey);
    }

    public static X509Certificate generateSelfsignedCert(X500Name subjectDn, SubjectPublicKeyInfo pubKeyInfo, PrivateKey identityKey) throws CertificateException {
        ContentSigner contentSigner;
        ScepUtil.requireNonNull("subjectDn", subjectDn);
        ScepUtil.requireNonNull("pubKeyInfo", pubKeyInfo);
        ScepUtil.requireNonNull("identityKey", identityKey);
        Date notBefore = new Date(System.currentTimeMillis() - 300000L);
        Date notAfter = new Date(notBefore.getTime() + 2592000000L);
        X509v3CertificateBuilder certGenerator = new X509v3CertificateBuilder(subjectDn, BigInteger.ONE, notBefore, notAfter, subjectDn, pubKeyInfo);
        X509KeyUsage ku = new X509KeyUsage(184);
        try {
            certGenerator.addExtension(Extension.keyUsage, true, (ASN1Encodable)ku);
        }
        catch (CertIOException ex) {
            throw new CertificateException("could not generate self-signed certificate: " + ex.getMessage(), ex);
        }
        String sigAlgorithm = ScepUtil.getSignatureAlgorithm(identityKey, ScepHashAlgoType.SHA1);
        try {
            contentSigner = new JcaContentSignerBuilder(sigAlgorithm).build(identityKey);
        }
        catch (OperatorCreationException ex) {
            throw new CertificateException("error while creating signer", ex);
        }
        Certificate asn1Cert = certGenerator.build(contentSigner).toASN1Structure();
        return ScepUtil.toX509Cert(asn1Cert);
    }

    public static List<X509Certificate> getCertsFromSignedData(SignedData signedData) throws CertificateException {
        ScepUtil.requireNonNull("signedData", signedData);
        ASN1Set set = signedData.getCertificates();
        if (set == null) {
            return Collections.emptyList();
        }
        int n = set.size();
        if (n == 0) {
            return Collections.emptyList();
        }
        LinkedList<X509Certificate> certs = new LinkedList<X509Certificate>();
        X509Certificate eeCert = null;
        for (int i = 0; i < n; ++i) {
            X509Certificate cert;
            try {
                cert = ScepUtil.toX509Cert(Certificate.getInstance((Object)set.getObjectAt(i)));
            }
            catch (IllegalArgumentException ex) {
                throw new CertificateException(ex);
            }
            if (eeCert == null && cert.getBasicConstraints() == -1) {
                eeCert = cert;
                continue;
            }
            certs.add(cert);
        }
        if (eeCert != null) {
            certs.add(0, eeCert);
        }
        return certs;
    }

    public static X509CRL getCrlFromPkiMessage(SignedData signedData) throws CRLException {
        ScepUtil.requireNonNull("signedData", signedData);
        ASN1Set set = signedData.getCRLs();
        if (set == null || set.size() == 0) {
            return null;
        }
        try {
            CertificateList cl = CertificateList.getInstance((Object)set.getObjectAt(0));
            return ScepUtil.toX509Crl(cl);
        }
        catch (IllegalArgumentException | CRLException | CertificateException ex) {
            throw new CRLException(ex);
        }
    }

    public static String getSignatureAlgorithm(PrivateKey key, ScepHashAlgoType hashAlgo) {
        ScepUtil.requireNonNull("key", key);
        ScepUtil.requireNonNull("hashAlgo", hashAlgo);
        String algorithm = key.getAlgorithm();
        if ("RSA".equalsIgnoreCase(algorithm)) {
            return hashAlgo.getName() + "withRSA";
        }
        throw new UnsupportedOperationException("getSignatureAlgorithm() for non-RSA is not supported yet.");
    }

    public static X509Certificate toX509Cert(Certificate asn1Cert) throws CertificateException {
        byte[] encodedCert;
        try {
            encodedCert = asn1Cert.getEncoded();
        }
        catch (IOException ex) {
            throw new CertificateEncodingException("could not get encoded certificate", ex);
        }
        return ScepUtil.parseCert(encodedCert);
    }

    public static X509CRL toX509Crl(CertificateList asn1CertList) throws CertificateException, CRLException {
        byte[] encodedCrl;
        try {
            encodedCrl = asn1CertList.getEncoded();
        }
        catch (IOException ex) {
            throw new CRLException("could not get encoded CRL", ex);
        }
        return ScepUtil.parseCrl(encodedCrl);
    }

    public static X509CRL parseCrl(byte[] encodedCrl) throws CertificateException, CRLException {
        ScepUtil.requireNonNull("encodedCrl", encodedCrl);
        return ScepUtil.parseCrl(new ByteArrayInputStream(encodedCrl));
    }

    public static X509CRL parseCrl(InputStream crlStream) throws CertificateException, CRLException {
        ScepUtil.requireNonNull("crlStream", crlStream);
        X509CRL crl = (X509CRL)ScepUtil.getCertFactory().generateCRL(crlStream);
        if (crl == null) {
            throw new CRLException("the given one is not a valid X.509 CRL");
        }
        return crl;
    }

    public static X509Certificate parseCert(byte[] certBytes) throws CertificateException {
        ScepUtil.requireNonNull("certBytes", certBytes);
        return ScepUtil.parseCert(new ByteArrayInputStream(certBytes));
    }

    private static X509Certificate parseCert(InputStream certStream) throws CertificateException {
        ScepUtil.requireNonNull("certStream", certStream);
        return (X509Certificate)ScepUtil.getCertFactory().generateCertificate(certStream);
    }

    private static byte[] extractSki(X509Certificate cert) throws CertificateEncodingException {
        byte[] extValue = ScepUtil.getCoreExtValue(cert, Extension.subjectKeyIdentifier);
        if (extValue == null) {
            return null;
        }
        try {
            return ASN1OctetString.getInstance((Object)extValue).getOctets();
        }
        catch (IllegalArgumentException ex) {
            throw new CertificateEncodingException(ex.getMessage());
        }
    }

    private static byte[] extractAki(X509Certificate cert) throws CertificateEncodingException {
        byte[] extValue = ScepUtil.getCoreExtValue(cert, Extension.authorityKeyIdentifier);
        if (extValue == null) {
            return null;
        }
        try {
            AuthorityKeyIdentifier aki = AuthorityKeyIdentifier.getInstance((Object)extValue);
            return aki.getKeyIdentifier();
        }
        catch (IllegalArgumentException ex) {
            throw new CertificateEncodingException("invalid extension AuthorityKeyIdentifier: " + ex.getMessage());
        }
    }

    public static boolean hasKeyusage(X509Certificate cert, KeyUsage usage) {
        boolean[] keyusage = cert.getKeyUsage();
        if (keyusage != null && keyusage.length > usage.bit()) {
            return keyusage[usage.bit()];
        }
        return false;
    }

    private static byte[] getCoreExtValue(X509Certificate cert, ASN1ObjectIdentifier type) throws CertificateEncodingException {
        ScepUtil.requireNonNull("cert", cert);
        ScepUtil.requireNonNull("type", type);
        byte[] fullExtValue = cert.getExtensionValue(type.getId());
        if (fullExtValue == null) {
            return null;
        }
        try {
            return ASN1OctetString.getInstance((Object)fullExtValue).getOctets();
        }
        catch (IllegalArgumentException ex) {
            throw new CertificateEncodingException("invalid extension " + type.getId() + ": " + ex.getMessage());
        }
    }

    public static boolean isSelfSigned(X509Certificate cert) {
        ScepUtil.requireNonNull("cert", cert);
        boolean equals = cert.getSubjectX500Principal().equals(cert.getIssuerX500Principal());
        if (!equals) {
            return false;
        }
        try {
            byte[] ski = ScepUtil.extractSki(cert);
            byte[] aki = ScepUtil.extractAki(cert);
            return ski != null && aki != null ? Arrays.equals(ski, aki) : true;
        }
        catch (CertificateEncodingException ex) {
            return false;
        }
    }

    public static boolean issues(X509Certificate issuerCert, X509Certificate cert) throws CertificateEncodingException {
        boolean isCa;
        ScepUtil.requireNonNull("issuerCert", issuerCert);
        ScepUtil.requireNonNull("cert", cert);
        boolean bl = isCa = issuerCert.getBasicConstraints() >= 0;
        if (!isCa) {
            return false;
        }
        boolean issues = issuerCert.getSubjectX500Principal().equals(cert.getIssuerX500Principal());
        if (issues) {
            byte[] ski = ScepUtil.extractSki(issuerCert);
            byte[] aki = ScepUtil.extractAki(cert);
            if (ski != null) {
                issues = Arrays.equals(ski, aki);
            }
        }
        if (issues) {
            long issuerNotBefore = issuerCert.getNotBefore().getTime();
            long issuerNotAfter = issuerCert.getNotAfter().getTime();
            long notBefore = cert.getNotBefore().getTime();
            issues = notBefore <= issuerNotAfter && notBefore >= issuerNotBefore;
        }
        return issues;
    }

    public static ASN1ObjectIdentifier extractDigesetAlgorithmIdentifier(String sigOid, byte[] sigParams) throws NoSuchAlgorithmException {
        ASN1ObjectIdentifier digestAlgOid;
        ScepUtil.requireNonBlank("sigOid", sigOid);
        ASN1ObjectIdentifier algOid = new ASN1ObjectIdentifier(sigOid);
        if (PKCSObjectIdentifiers.md5WithRSAEncryption.equals((Object)algOid)) {
            digestAlgOid = PKCSObjectIdentifiers.md5;
        } else if (PKCSObjectIdentifiers.sha1WithRSAEncryption.equals((Object)algOid)) {
            digestAlgOid = X509ObjectIdentifiers.id_SHA1;
        } else if (PKCSObjectIdentifiers.sha224WithRSAEncryption.equals((Object)algOid)) {
            digestAlgOid = NISTObjectIdentifiers.id_sha224;
        } else if (PKCSObjectIdentifiers.sha256WithRSAEncryption.equals((Object)algOid)) {
            digestAlgOid = NISTObjectIdentifiers.id_sha256;
        } else if (PKCSObjectIdentifiers.sha384WithRSAEncryption.equals((Object)algOid)) {
            digestAlgOid = NISTObjectIdentifiers.id_sha384;
        } else if (PKCSObjectIdentifiers.sha512WithRSAEncryption.equals((Object)algOid)) {
            digestAlgOid = NISTObjectIdentifiers.id_sha512;
        } else if (PKCSObjectIdentifiers.id_RSASSA_PSS.equals((Object)algOid)) {
            RSASSAPSSparams param = RSASSAPSSparams.getInstance((Object)sigParams);
            digestAlgOid = param.getHashAlgorithm().getAlgorithm();
        } else {
            throw new NoSuchAlgorithmException("unknown signature algorithm" + algOid.getId());
        }
        return digestAlgOid;
    }

    public static ASN1Encodable getFirstAttrValue(AttributeTable attrs, ASN1ObjectIdentifier type) {
        ScepUtil.requireNonNull("attrs", attrs);
        ScepUtil.requireNonNull("type", type);
        Attribute attr = attrs.get(type);
        if (attr == null) {
            return null;
        }
        ASN1Set set = attr.getAttrValues();
        return set.size() == 0 ? null : set.getObjectAt(0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static byte[] read(InputStream in) throws IOException {
        ScepUtil.requireNonNull("in", in);
        try {
            ByteArrayOutputStream bout = new ByteArrayOutputStream();
            int readed = 0;
            byte[] buffer = new byte[2048];
            while ((readed = in.read(buffer)) != -1) {
                bout.write(buffer, 0, readed);
            }
            byte[] byArray = bout.toByteArray();
            return byArray;
        }
        finally {
            try {
                in.close();
            }
            catch (IOException ex) {
                LOG.error("could not close stream: {}", (Object)ex.getMessage());
            }
        }
    }

    public static void addCmsCertSet(CMSSignedDataGenerator generator, X509Certificate[] cmsCertSet) throws CertificateEncodingException, CMSException {
        if (cmsCertSet == null || cmsCertSet.length == 0) {
            return;
        }
        ScepUtil.requireNonNull("geneator", generator);
        LinkedList<X509Certificate> certColl = new LinkedList<X509Certificate>();
        for (X509Certificate m : cmsCertSet) {
            certColl.add(m);
        }
        JcaCertStore certStore = new JcaCertStore(certColl);
        generator.addCertificates((Store)certStore);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static CertificateFactory getCertFactory() throws CertificateException {
        Object object = certFactLock;
        synchronized (object) {
            if (certFact == null) {
                certFact = CertificateFactory.getInstance("X.509");
            }
            return certFact;
        }
    }

    public static <T> T requireNonNull(String objName, T obj) {
        return Objects.requireNonNull(obj, objName + " must not be null");
    }

    public static String requireNonBlank(String objName, String obj) {
        Objects.requireNonNull(obj, objName + " must not be null");
        if (obj.isEmpty()) {
            throw new IllegalArgumentException(objName + " must not be blank");
        }
        return obj;
    }

    public static <T> Collection<T> requireNonEmpty(String objName, Collection<T> obj) {
        Objects.requireNonNull(obj, objName + " must not be null");
        if (obj.isEmpty()) {
            throw new IllegalArgumentException(objName + " must not be empty");
        }
        return obj;
    }

    static {
        certFactLock = new Object();
    }
}

