/*
 * Decompiled with CFR 0.152.
 */
package org.xsocket;

import java.io.IOException;
import java.lang.ref.SoftReference;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import org.xsocket.ClosedConnectionException;
import org.xsocket.Connection;
import org.xsocket.SSLClientProcessor;
import org.xsocket.SSLServerProcessor;
import org.xsocket.util.TextUtils;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
abstract class SSLProcessor {
    private static final Logger LOG = Logger.getLogger(SSLProcessor.class.getName());
    private Connection connection = null;
    private SSLEngine sslEngine = null;
    private SSLMemoryManager memoryManager = new SSLMemoryManager(65536);
    private int minPacketBufferSize = 0;
    private int minAppBufferSize = 0;

    SSLProcessor(Connection connection, boolean clientMode, SSLContext sslContext) {
        this.connection = connection;
        this.sslEngine = sslContext.createSSLEngine();
        this.minAppBufferSize = this.sslEngine.getSession().getApplicationBufferSize();
        this.minPacketBufferSize = this.sslEngine.getSession().getPacketBufferSize();
        this.sslEngine.setUseClientMode(clientMode);
    }

    static final SSLProcessor newProcessor(Connection connection, boolean clientMode, SSLContext sslContext) {
        if (clientMode) {
            return new SSLClientProcessor(connection, sslContext);
        }
        return new SSLServerProcessor(connection, sslContext);
    }

    protected final Connection getConnection() {
        return this.connection;
    }

    void start() throws IOException {
        try {
            this.sslEngine.beginHandshake();
        }
        catch (SSLException sslEx) {
            throw new RuntimeException(sslEx);
        }
    }

    final void destroy() {
        this.sslEngine.closeOutbound();
    }

    abstract void writeOutgoing(ByteBuffer[] var1) throws ClosedConnectionException, IOException;

    abstract int readIncoming() throws IOException, ClosedConnectionException;

    final int read() throws IOException, ClosedConnectionException {
        int readSize = 0;
        ByteBuffer inNetbuffer = this.connection.readPhysical();
        if (inNetbuffer != null) {
            List<ByteBuffer> inAppDataList = this.unwrap(inNetbuffer);
            for (ByteBuffer inAppData : inAppDataList) {
                if (inAppData.remaining() <= 0) continue;
                if (LOG.isLoggable(Level.FINE)) {
                    this.getConnection().logFine("ssl -> data: " + TextUtils.toString(new ByteBuffer[]{inAppData.duplicate()}, "UTF-8", 500));
                }
                readSize += this.connection.appenToReadQueue(inAppData);
            }
        }
        return readSize;
    }

    final void write(ByteBuffer[] buffers) throws ClosedConnectionException, IOException {
        if (buffers.length > 0) {
            for (ByteBuffer outAppData : buffers) {
                List<ByteBuffer> outNetDataList;
                if (LOG.isLoggable(Level.FINE) && outAppData.remaining() > 0) {
                    this.getConnection().logFine("data -> ssl: " + TextUtils.toString(new ByteBuffer[]{outAppData.duplicate()}, "UTF-8", 500));
                }
                if ((outNetDataList = this.wrap(outAppData)).size() <= 0) continue;
                ByteBuffer[] outBuffers = outNetDataList.toArray(new ByteBuffer[outNetDataList.size()]);
                this.getConnection().writePhysical(outBuffers);
            }
        }
    }

    final boolean needWrap() {
        return this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_WRAP;
    }

    final boolean needUnwrap() {
        return this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_UNWRAP;
    }

    final boolean isHandshaking() {
        return this.sslEngine.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING;
    }

    final List<ByteBuffer> unwrap(ByteBuffer inNetData) throws SSLException, ClosedConnectionException {
        ArrayList<ByteBuffer> inAppDataList = new ArrayList<ByteBuffer>();
        boolean repeat = false;
        int minSize = this.minAppBufferSize;
        try {
            do {
                repeat = false;
                SSLEngineResult engineResult = null;
                ByteBuffer inAppData = this.memoryManager.acquireMemory(minSize);
                engineResult = this.sslEngine.unwrap(inNetData, inAppData);
                if (LOG.isLoggable(Level.FINER)) {
                    LOG.finer(engineResult.toString());
                }
                if (engineResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                    LOG.warning("BUFFER_UNDERFLOW within unwrap shouldn't occur");
                } else {
                    if (engineResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                        this.memoryManager.recycleMemory(inAppData);
                        repeat = true;
                        minSize += minSize;
                        continue;
                    }
                    if (engineResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                        if (LOG.isLoggable(Level.FINER)) {
                            LOG.finer("ssl connection is closed. closing connection");
                        }
                        this.memoryManager.recycleMemory(inAppData);
                        ClosedConnectionException cce = new ClosedConnectionException("Couldn't unwrap, because connection is closed");
                        LOG.throwing(this.getClass().getName(), "unwrap", cce);
                        throw cce;
                    }
                    if (engineResult.getStatus() == SSLEngineResult.Status.OK) {
                        if (inNetData.position() < inNetData.limit()) {
                            inNetData = inNetData.slice();
                        }
                        inAppData.flip();
                        inAppData = this.memoryManager.extractAndRecycleMemory(inAppData);
                        if (inAppData.remaining() > 0) {
                            inAppDataList.add(inAppData);
                        }
                    }
                }
                if (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
                    Runnable task = null;
                    while ((task = this.sslEngine.getDelegatedTask()) != null) {
                        task.run();
                    }
                    if (LOG.isLoggable(Level.FINER)) {
                        LOG.finer(this.sslEngine.getHandshakeStatus().toString());
                    }
                }
                if (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_UNWRAP && inNetData.hasRemaining()) {
                    repeat = true;
                }
                if (engineResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.FINISHED || this.connection.isConnected()) continue;
                if (LOG.isLoggable(Level.FINE)) {
                    LOG.fine("handshake finished (client side)");
                }
                this.connection.onConnect();
            } while (repeat);
        }
        catch (SSLException ssles) {
            ssles.printStackTrace();
            throw ssles;
        }
        return inAppDataList;
    }

    final List<ByteBuffer> wrap(ByteBuffer outAppData) throws SSLException, ClosedConnectionException, IOException {
        ArrayList<ByteBuffer> outNetDataList = new ArrayList<ByteBuffer>();
        boolean repeat = false;
        int minSize = this.minPacketBufferSize;
        do {
            repeat = false;
            SSLEngineResult engineResult = null;
            ByteBuffer outNetData = this.memoryManager.acquireMemory(minSize);
            engineResult = this.sslEngine.wrap(outAppData, outNetData);
            if (LOG.isLoggable(Level.FINER)) {
                LOG.finer(engineResult.toString());
            }
            if (engineResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                this.memoryManager.recycleMemory(outNetData);
                repeat = true;
                minSize += minSize;
                continue;
            }
            if (engineResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                if (LOG.isLoggable(Level.FINER)) {
                    LOG.finer("ssl connection is closed. closing connection");
                }
                this.memoryManager.recycleMemory(outNetData);
                ClosedConnectionException cce = new ClosedConnectionException("Couldn't unwrap, because connection is closed");
                LOG.throwing(this.getClass().getName(), "wrap", cce);
                throw cce;
            }
            if (engineResult.getStatus() == SSLEngineResult.Status.OK) {
                if (outAppData.position() < outAppData.limit()) {
                    outAppData = outAppData.slice();
                }
                outNetData.flip();
                outNetData = this.memoryManager.extractAndRecycleMemory(outNetData);
                outNetDataList.add(outNetData);
            }
            if (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
                Runnable task = null;
                while ((task = this.sslEngine.getDelegatedTask()) != null) {
                    task.run();
                }
                if (LOG.isLoggable(Level.FINER)) {
                    LOG.finer(this.sslEngine.getHandshakeStatus().toString());
                }
            }
            if (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_WRAP) {
                repeat = true;
            }
            if (engineResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.FINISHED) continue;
            if (this.connection.isConnected()) break;
            if (LOG.isLoggable(Level.FINE)) {
                LOG.fine("handshake finished (server side)");
            }
            this.connection.onConnect();
            break;
        } while (repeat);
        return outNetDataList;
    }

    private static final class SSLMemoryManager {
        private List<SoftReference<ByteBuffer>> memoryBuffer = new ArrayList<SoftReference<ByteBuffer>>();
        private int preallocationSize = 4096;

        SSLMemoryManager(int preallocationSize) {
            this.preallocationSize = preallocationSize;
        }

        public synchronized void recycleMemory(ByteBuffer buffer) {
            assert (buffer.position() == 0) : "buffer must be clean";
            this.memoryBuffer.add(new SoftReference<ByteBuffer>(buffer));
        }

        public synchronized ByteBuffer acquireMemory(int minSize) {
            SoftReference<ByteBuffer> freeBuffer;
            ByteBuffer buffer = null;
            if (!this.memoryBuffer.isEmpty() && (buffer = (freeBuffer = this.memoryBuffer.remove(0)).get()) != null && buffer.limit() < minSize) {
                buffer = null;
            }
            if (buffer == null) {
                int size = this.preallocationSize;
                if (size < minSize) {
                    size = minSize * 4;
                }
                buffer = ByteBuffer.allocateDirect(size);
            }
            return buffer;
        }

        public ByteBuffer extractAndRecycleMemory(ByteBuffer buffer) {
            if (buffer.limit() == buffer.capacity()) {
                return buffer;
            }
            int savedLimit = buffer.limit();
            ByteBuffer slicedPart = buffer.slice();
            buffer.position(savedLimit);
            buffer.limit(buffer.capacity());
            ByteBuffer unused = buffer.slice();
            this.recycleMemory(unused);
            return slicedPart;
        }
    }
}

