/*
 * Decompiled with CFR 0.152.
 */
package org.xsocket.stream.io.impl;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import org.xsocket.ClosedConnectionException;
import org.xsocket.DataConverter;
import org.xsocket.stream.io.impl.IMemoryManager;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
final class SSLProcessor {
    private static final Logger LOG = Logger.getLogger(SSLProcessor.class.getName());
    private SSLEngine sslEngine = null;
    private boolean isClientMode = false;
    private IMemoryManager memoryManager = null;
    private EventHandler eventHandler = null;
    private int minNetBufferSize = 0;
    private int minEncryptedBufferSize = 0;
    private ByteBuffer unprocessedEncryptedData = null;

    SSLProcessor(SSLContext sslContext, boolean isClientMode, IMemoryManager memoryManager, EventHandler eventHandler) {
        this.isClientMode = isClientMode;
        this.memoryManager = memoryManager;
        this.eventHandler = eventHandler;
        this.sslEngine = sslContext.createSSLEngine();
        this.minEncryptedBufferSize = this.sslEngine.getSession().getApplicationBufferSize();
        this.minNetBufferSize = this.sslEngine.getSession().getPacketBufferSize();
        if (LOG.isLoggable(Level.FINE)) {
            if (isClientMode) {
                LOG.fine("initializing ssl processor (client mode)");
            } else {
                LOG.fine("initializing ssl processor (server mode)");
            }
            LOG.fine("app buffer size is " + this.minEncryptedBufferSize);
            LOG.fine("packet buffer size is " + this.minNetBufferSize);
        }
        this.sslEngine.setUseClientMode(isClientMode);
    }

    synchronized void start() throws IOException {
        try {
            this.sslEngine.beginHandshake();
        }
        catch (SSLException sslEx) {
            throw new RuntimeException(sslEx);
        }
        if (this.isClientMode) {
            if (LOG.isLoggable(Level.FINE)) {
                LOG.fine("initate ssl handshake");
            }
            this.encrypt();
        }
    }

    void destroy() {
        this.sslEngine.closeOutbound();
    }

    synchronized void decrypt(LinkedList<ByteBuffer> encryptedBufferList) throws IOException, ClosedConnectionException {
        if (LOG.isLoggable(Level.FINEST)) {
            LOG.finest("decrypting " + encryptedBufferList.size() + " buffers");
        }
        LinkedList<ByteBuffer> decryptedDataList = new LinkedList<ByteBuffer>();
        try {
            for (int i = 0; i < encryptedBufferList.size(); ++i) {
                if (LOG.isLoggable(Level.FINEST)) {
                    LOG.finest("processing " + i + ".buffer (encrypted size " + encryptedBufferList.get(i).remaining() + ")");
                }
                LinkedList<ByteBuffer> inAppData = this.unwrap(encryptedBufferList.get(i));
                decryptedDataList.addAll(inAppData);
                if (!LOG.isLoggable(Level.FINE)) continue;
                int decryptedDataSize = 0;
                ByteBuffer[] decryptedDataListCopy = new ByteBuffer[decryptedDataList.size()];
                for (int j = 0; j < decryptedDataList.size(); ++j) {
                    decryptedDataSize += decryptedDataList.get(j).remaining();
                    decryptedDataListCopy[j] = decryptedDataList.get(j).duplicate();
                }
                if (decryptedDataSize <= 0) continue;
                LOG.fine(decryptedDataSize + " decrypted data: " + DataConverter.toTextOrHexString(decryptedDataListCopy, "UTF-8", 500));
            }
            if (!decryptedDataList.isEmpty()) {
                this.eventHandler.onDataDecrypted(decryptedDataList);
            }
        }
        catch (SSLException sslEx) {
            this.eventHandler.onSSLProcessorClosed();
        }
        if (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_WRAP) {
            this.encrypt();
        }
    }

    synchronized boolean isHandshaking() {
        SSLEngineResult.HandshakeStatus handshakeStatus = this.sslEngine.getHandshakeStatus();
        return handshakeStatus != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING || handshakeStatus == SSLEngineResult.HandshakeStatus.FINISHED;
    }

    void encrypt(LinkedList<ByteBuffer> plainBufferList) throws ClosedConnectionException, IOException {
        for (ByteBuffer plainBuffer : plainBufferList) {
            this.wrap(plainBuffer);
        }
    }

    void encrypt() throws ClosedConnectionException, IOException {
        ByteBuffer outAppData = ByteBuffer.allocate(0);
        this.wrap(outAppData);
    }

    private synchronized LinkedList<ByteBuffer> unwrap(ByteBuffer inNetData) throws SSLException, ClosedConnectionException, IOException {
        LinkedList<ByteBuffer> inAppDataList = new LinkedList<ByteBuffer>();
        int minAppSize = this.minEncryptedBufferSize;
        if (this.unprocessedEncryptedData != null) {
            if (LOG.isLoggable(Level.FINEST)) {
                LOG.finest("unprocessed encrypted data available. merging unprocessed encrypted data (size " + this.unprocessedEncryptedData.remaining() + ") with " + " new encrypted data (size " + inNetData.remaining() + ")");
            }
            inNetData = this.mergeBuffer(this.unprocessedEncryptedData, inNetData);
            this.unprocessedEncryptedData = null;
            if (LOG.isLoggable(Level.FINEST)) {
                LOG.finest("new encrypted data size is " + inNetData.remaining());
            }
        }
        do {
            ByteBuffer inAppData;
            SSLEngineResult engineResult;
            if ((engineResult = this.sslEngine.unwrap(inNetData, inAppData = this.memoryManager.acquireMemory(minAppSize))).getStatus() == SSLEngineResult.Status.OK) {
                if (inNetData.position() < inNetData.limit()) {
                    inNetData = inNetData.slice();
                }
                inAppData.flip();
                inAppData = this.extractAndRecycleMemory(inAppData, minAppSize);
                if (inAppData.remaining() > 0) {
                    inAppDataList.add(inAppData);
                    if (LOG.isLoggable(Level.FINEST)) {
                        LOG.finest(inAppData.remaining() + " plain data has decrypted");
                    }
                } else if (LOG.isLoggable(Level.FINEST)) {
                    LOG.finest("SSL System data (InNetData doesn't contain app data)");
                }
            } else {
                if (engineResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                    if (LOG.isLoggable(Level.FINEST)) {
                        LOG.finest("BufferUnderflow occured (not enough InNet data)");
                    }
                    this.unprocessedEncryptedData = inNetData;
                    break;
                }
                if (engineResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                    this.memoryManager.recycleMemory(inAppData, minAppSize);
                    minAppSize += minAppSize;
                    continue;
                }
                if (engineResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                    this.memoryManager.recycleMemory(inAppData, minAppSize);
                    throw new ClosedConnectionException("Couldn't unwrap, because connection is closed");
                }
            }
            if (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
                Runnable task = null;
                while ((task = this.sslEngine.getDelegatedTask()) != null) {
                    task.run();
                }
            }
            if (engineResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.FINISHED) continue;
            this.notifyHandshakeFinished();
        } while (inNetData.hasRemaining());
        return inAppDataList;
    }

    private synchronized void wrap(ByteBuffer outAppData) throws SSLException, ClosedConnectionException, IOException {
        int minNetSize = this.minNetBufferSize;
        do {
            ByteBuffer outNetData;
            SSLEngineResult engineResult;
            if ((engineResult = this.sslEngine.wrap(outAppData, outNetData = this.memoryManager.acquireMemory(minNetSize))).getStatus() == SSLEngineResult.Status.OK || engineResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                if (outAppData.position() < outAppData.limit()) {
                    outAppData = outAppData.slice();
                }
                outNetData.flip();
                outNetData = this.extractAndRecycleMemory(outNetData, minNetSize);
                if (outNetData.hasRemaining()) {
                    this.eventHandler.onDataEncrypted(outNetData);
                }
            } else {
                if (engineResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                    this.memoryManager.recycleMemory(outNetData, minNetSize);
                    minNetSize += minNetSize;
                    continue;
                }
                if (engineResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                    this.memoryManager.recycleMemory(outNetData, minNetSize);
                    throw new ClosedConnectionException("Couldn't wrap, because connection is closed");
                }
            }
            if (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
                Runnable task = null;
                while ((task = this.sslEngine.getDelegatedTask()) != null) {
                    task.run();
                }
            }
            if (engineResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.FINISHED) continue;
            this.notifyHandshakeFinished();
            break;
        } while (outAppData.hasRemaining() || this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_WRAP);
    }

    private void notifyHandshakeFinished() throws IOException {
        if (LOG.isLoggable(Level.FINE)) {
            if (this.isClientMode) {
                LOG.fine("handshake has been finished (clientMode)");
            } else {
                LOG.fine("handshake has been finished (serverMode)");
            }
        }
        this.encrypt();
        this.eventHandler.onHandshakeFinished();
    }

    private ByteBuffer mergeBuffer(ByteBuffer first, ByteBuffer second) {
        ByteBuffer mergedBuffer = ByteBuffer.allocate(first.remaining() + second.remaining());
        mergedBuffer.put(first);
        mergedBuffer.put(second);
        mergedBuffer.flip();
        return mergedBuffer;
    }

    private ByteBuffer extractAndRecycleMemory(ByteBuffer buffer, int minSize) {
        if (buffer.limit() == buffer.capacity()) {
            return buffer;
        }
        int savedLimit = buffer.limit();
        ByteBuffer slicedPart = buffer.slice();
        buffer.position(savedLimit);
        buffer.limit(buffer.capacity());
        ByteBuffer unused = buffer.slice();
        this.memoryManager.recycleMemory(unused, minSize);
        return slicedPart;
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    static interface EventHandler {
        public void onHandshakeFinished() throws IOException;

        public void onDataDecrypted(LinkedList<ByteBuffer> var1);

        public void onDataEncrypted(ByteBuffer var1) throws IOException;

        public void onSSLProcessorClosed() throws IOException;
    }
}

