/*
 * Decompiled with CFR 0.152.
 */
package org.xsocket.stream.io.impl;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import org.xsocket.ClosedConnectionException;
import org.xsocket.DataConverter;
import org.xsocket.stream.io.impl.IMemoryManager;

final class SSLProcessor {
    private static final Logger LOG = Logger.getLogger(SSLProcessor.class.getName());
    private SSLEngine sslEngine = null;
    private boolean isClientMode = false;
    private IMemoryManager memoryManager = null;
    private EventHandler eventHandler = null;
    private int minNetBufferSize = 0;
    private int minAppBufferSize = 0;

    SSLProcessor(SSLContext sslContext, boolean isClientMode, IMemoryManager memoryManager, EventHandler eventHandler) {
        this.isClientMode = isClientMode;
        this.memoryManager = memoryManager;
        this.eventHandler = eventHandler;
        this.sslEngine = sslContext.createSSLEngine();
        this.minAppBufferSize = this.sslEngine.getSession().getApplicationBufferSize();
        this.minNetBufferSize = this.sslEngine.getSession().getPacketBufferSize();
        this.sslEngine.setUseClientMode(isClientMode);
    }

    synchronized void start() throws IOException {
        try {
            this.sslEngine.beginHandshake();
        }
        catch (SSLException sslEx) {
            throw new RuntimeException(sslEx);
        }
        if (this.isClientMode) {
            if (LOG.isLoggable(Level.FINE)) {
                LOG.fine("initate ssl handshake");
            }
            this.processOutAppData();
        }
    }

    void destroy() {
        this.sslEngine.closeOutbound();
    }

    synchronized void processInNetData(LinkedList<ByteBuffer> inNetBufferList) throws IOException, ClosedConnectionException {
        LinkedList<ByteBuffer> inAppDataList = new LinkedList<ByteBuffer>();
        try {
            int size = 0;
            for (ByteBuffer buffer : inNetBufferList) {
                size += buffer.remaining();
            }
            for (ByteBuffer inNetBuffer : inNetBufferList) {
                LinkedList<ByteBuffer> inAppData = this.unwrap(inNetBuffer);
                inAppDataList.addAll(inAppData);
                if (!LOG.isLoggable(Level.FINE)) continue;
                int appDataSize = 0;
                ByteBuffer[] inAppDataListCopy = new ByteBuffer[inAppDataList.size()];
                for (int i = 0; i < inAppDataList.size(); ++i) {
                    appDataSize += inAppDataList.get(i).remaining();
                    inAppDataListCopy[i] = inAppDataList.get(i).duplicate();
                }
                if (size - appDataSize > 0) {
                    LOG.fine(size - appDataSize + " ssl system packet data ");
                }
                if (appDataSize <= 0) continue;
                LOG.fine(appDataSize + " app data extracted: " + DataConverter.toTextOrHexString(inAppDataListCopy, "UTF-8", 500));
            }
            if (!inAppDataList.isEmpty()) {
                this.eventHandler.onInAppDataReceived(inAppDataList);
            }
        }
        catch (SSLException sslEx) {
            this.eventHandler.onSSLProcessorClosed();
        }
        if (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_WRAP) {
            this.processOutAppData();
        }
    }

    synchronized boolean isHandshaking() {
        SSLEngineResult.HandshakeStatus handshakeStatus = this.sslEngine.getHandshakeStatus();
        return handshakeStatus != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING || handshakeStatus == SSLEngineResult.HandshakeStatus.FINISHED;
    }

    void processOutAppData(LinkedList<ByteBuffer> outAppDataList) throws ClosedConnectionException, IOException {
        for (ByteBuffer outAppData : outAppDataList) {
            this.wrap(outAppData);
        }
    }

    void processOutAppData() throws ClosedConnectionException, IOException {
        ByteBuffer outAppData = ByteBuffer.allocate(0);
        this.wrap(outAppData);
    }

    private synchronized LinkedList<ByteBuffer> unwrap(ByteBuffer inNetData) throws SSLException, ClosedConnectionException, IOException {
        LinkedList<ByteBuffer> inAppDataList = new LinkedList<ByteBuffer>();
        int minAppSize = this.minAppBufferSize;
        do {
            ByteBuffer inAppData;
            SSLEngineResult engineResult;
            if ((engineResult = this.sslEngine.unwrap(inNetData, inAppData = this.memoryManager.acquireMemory(minAppSize))).getStatus() == SSLEngineResult.Status.OK || engineResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                if (inNetData.position() < inNetData.limit()) {
                    inNetData = inNetData.slice();
                }
                inAppData.flip();
                inAppData = this.extractAndRecycleMemory(inAppData, minAppSize);
                if (inAppData.remaining() > 0) {
                    inAppDataList.add(inAppData);
                }
            } else {
                if (engineResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                    this.memoryManager.recycleMemory(inAppData, minAppSize);
                    minAppSize += minAppSize;
                    continue;
                }
                if (engineResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                    this.memoryManager.recycleMemory(inAppData, minAppSize);
                    throw new ClosedConnectionException("Could not unwrap, because connection is closed");
                }
            }
            if (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
                Runnable task = null;
                while ((task = this.sslEngine.getDelegatedTask()) != null) {
                    task.run();
                }
            }
            if (engineResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.FINISHED) continue;
            this.notifyHandshakeFinished();
        } while (inNetData.hasRemaining());
        return inAppDataList;
    }

    private synchronized void wrap(ByteBuffer outAppData) throws SSLException, ClosedConnectionException, IOException {
        int minNetSize = this.minNetBufferSize;
        do {
            ByteBuffer outNetData;
            SSLEngineResult engineResult;
            if ((engineResult = this.sslEngine.wrap(outAppData, outNetData = this.memoryManager.acquireMemory(minNetSize))).getStatus() == SSLEngineResult.Status.OK || engineResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                if (outAppData.position() < outAppData.limit()) {
                    outAppData = outAppData.slice();
                }
                outNetData.flip();
                outNetData = this.extractAndRecycleMemory(outNetData, minNetSize);
                if (outNetData.hasRemaining()) {
                    this.eventHandler.onOutNetDataToWrite(outNetData);
                }
            } else {
                if (engineResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                    this.memoryManager.recycleMemory(outNetData, minNetSize);
                    minNetSize += minNetSize;
                    continue;
                }
                if (engineResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                    this.memoryManager.recycleMemory(outNetData, minNetSize);
                    throw new ClosedConnectionException("Could not wrap, because connection is closed");
                }
            }
            if (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
                Runnable task = null;
                while ((task = this.sslEngine.getDelegatedTask()) != null) {
                    task.run();
                }
            }
            if (engineResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.FINISHED) continue;
            this.notifyHandshakeFinished();
            break;
        } while (outAppData.hasRemaining() || this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_WRAP);
    }

    private void notifyHandshakeFinished() throws IOException {
        if (LOG.isLoggable(Level.FINE)) {
            if (this.isClientMode) {
                LOG.fine("handshake has been finished (clientMode)");
            } else {
                LOG.fine("handshake has been finished (serverMode)");
            }
        }
        this.processOutAppData();
        this.eventHandler.onHandshakeFinished();
    }

    private ByteBuffer extractAndRecycleMemory(ByteBuffer buffer, int minSize) {
        if (buffer.limit() == buffer.capacity()) {
            return buffer;
        }
        int savedLimit = buffer.limit();
        ByteBuffer slicedPart = buffer.slice();
        buffer.position(savedLimit);
        buffer.limit(buffer.capacity());
        ByteBuffer unused = buffer.slice();
        this.memoryManager.recycleMemory(unused, minSize);
        return slicedPart;
    }

    static interface EventHandler {
        public void onHandshakeFinished() throws IOException;

        public void onInAppDataReceived(LinkedList<ByteBuffer> var1);

        public void onOutNetDataToWrite(ByteBuffer var1) throws IOException;

        public void onSSLProcessorClosed() throws IOException;
    }
}

