/*
 *  Copyright (c) xsocket.org, 2006 - 2009. All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Please refer to the LGPL license at: http://www.gnu.org/copyleft/lesser.txt
 * The latest copy of this software may be found on http://www.xsocket.org/
 */
package org.xsocket.connection;


import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketTimeoutException;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TimerTask;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.net.ssl.SSLContext;

import org.xsocket.DataConverter;
import org.xsocket.Execution;
import org.xsocket.ILifeCycle;
import org.xsocket.MaxReadSizeExceededException;





/**
 * A connection pool implementation.<br> <br>
 *
 * This class is thread safe <br>
 *
 * <pre>
 *  // create a unbound connection pool
 *  NonBlockingConnectionPool pool = new NonBlockingConnectionPool();
 *
 *  INonBlockingCinnection con = null;
 *
 *  try {
 *     // retrieve a connection (if no connection is in pool, a new one will be created)
 *     con = pool.getNonBlockingConnection(host, port);
 *     con.write("Hello");
 *     ...
 *
 *     // always close the connection! (the connection will be returned into the connection pool)
 *     con.close();
 *
 * 	} catch (IOException e) {
 *     if (con != null) {
 *        try {
 *          // if the connection is invalid -> destroy it (it will not return into the pool)
 *          pool.destroy(con);
 *        } catch (Exception ignore) { }
 *     }
 *  }
 * </pre>
 *
 * @author grro@xsocket.org
 */
public final class NonBlockingConnectionPool implements IConnectionPool {

	private static final Logger LOG = Logger.getLogger(NonBlockingConnectionPool.class.getName());
	
	
	private static final long MIN_REMAINING_MILLIS_TO_IDLE_TIMEOUT = 3 * 1000;
	private static final long MIN_REMAINING_MILLIS_TO_CONNECTION_TIMEOUT = 3 * 1000;


	// is open flag
	private final AtomicBoolean isOpen = new AtomicBoolean(true);

		
	// ssl support
	private final SSLContext sslContext;

	
	// settings
	private final AtomicInteger maxActive = new AtomicInteger(IConnectionPool.DEFAULT_MAX_ACTIVE);
	private final AtomicInteger maxIdle = new AtomicInteger(IConnectionPool.DEFAULT_MAX_IDLE);
	private final AtomicLong maxWaitMillis = new AtomicLong(IConnectionPool.DEFAULT_MAX_WAIT_MILLIS);
	private final AtomicInteger poolIdleTimeoutMillis = new AtomicInteger(IConnectionPool.DEFAULT_IDLE_TIMEOUT_MILLIS);
	private final AtomicInteger lifeTimeoutMillis = new AtomicInteger(IConnectionPool.DEFAULT_LIFE_TIMEOUT_MILLIS);
	

	// pool management
	private final Pool pool = new Pool();
	private final Object retrieveGuard = new Object();
	private Executor workerpool = NonBlockingConnection.getDefaultWorkerpool();

	
	// timeout checker 
	private int watchdogCheckPeriod = Integer.MAX_VALUE;
	private Watchog watchdog = null;
	
	
	// listeners
	private final List<ILifeCycle> listeners = new ArrayList<ILifeCycle>();


	// statistics
	private final AtomicInteger countPendingGet = new AtomicInteger(0);
	private final AtomicInteger countCreated = new AtomicInteger(0);
	private final AtomicInteger countDestroyed = new AtomicInteger(0);
	private final AtomicInteger countRemainingMillisToIdleTimeoutToSmall = new AtomicInteger(0);
	private final AtomicInteger countRemainingConnectionToIdleTimeoutToSmall = new AtomicInteger(0);
	private final AtomicInteger countCreationError = new AtomicInteger(0);
	private final AtomicInteger countIdleTimeout = new AtomicInteger(0);
	private final AtomicInteger countConnectionTimeout = new AtomicInteger(0);
	private final AtomicInteger countTimeoutPooledIdle = new AtomicInteger(0);
	private final AtomicInteger countTimeoutPooledLifetime = new AtomicInteger(0);
	
	

	/**
	 * constructor
	 *
	 */
	public NonBlockingConnectionPool() {
	    this(null);
	}

	
	/**
	 * constructor
	 *
	 * @param sslContext   the ssl context or <code>null</code> if ssl should not be used
	 */
	public NonBlockingConnectionPool(SSLContext sslContext) {
		this.sslContext = sslContext;
	}



	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created
	 *
	 * @param host   the server address
	 * @param port   the server port
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(String host, int port) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(host, port), null, workerpool, Integer.MAX_VALUE, maxWaitMillis.get(), false);
	}




	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created
	 *
	 * @param host   the server address
	 * @param port   the server port
	 * @param isSSL  true, if ssl connection

	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(String host, int port, boolean isSSL) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(host, port), null, workerpool, Integer.MAX_VALUE, maxWaitMillis.get(), isSSL);
	}


	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created
	 *
	 * @param host                   the server address
	 * @param port                   the server port
     * @param connectTimeoutMillis   the connection timeout
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(String host, int port, int connectTimeoutMillis) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(host, port), null, workerpool, connectTimeoutMillis, maxWaitMillis.get(), false);
	}


	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created
	 *
	 * @param host                   the server address
	 * @param port                   the server port
     * @param connectTimeoutMillis   the connection timeout
	 * @param isSSL                  true, if ssl connection
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(String host, int port, int connectTimeoutMillis, boolean isSSL) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(host, port), null, workerpool, connectTimeoutMillis, maxWaitMillis.get(), isSSL);
	}

	
	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created
	 * 
	 * @param address  the server address
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetSocketAddress address) throws IOException, SocketTimeoutException {
		return getConnection(address, null, workerpool, Integer.MAX_VALUE, maxWaitMillis.get(), false);
	}


	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created
	 *
	 * @param address  the server address
	 * @param port     the sever port
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetAddress address, int port) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(address, port), null, workerpool, Integer.MAX_VALUE, maxWaitMillis.get(), false);
	}


	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created
	 *
	 * @param address  the server address
	 * @param port     the sever port
	 * @param isSSL    true, if ssl connection
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetAddress address, int port, boolean isSSL) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(address, port), null, workerpool, Integer.MAX_VALUE, maxWaitMillis.get(), isSSL);
	}

	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created
	 *
	 * @param address                the server address
	 * @param port                   the server port
     * @param connectTimeoutMillis   the connection timeout
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetAddress address, int port, int connectTimeoutMillis) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(address, port), null, workerpool, connectTimeoutMillis, maxWaitMillis.get(), false);
	}



	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created
	 *
	 * @param address                the server address
	 * @param port                   the server port
     * @param connectTimeoutMillis   the connection timeout
	 * @param isSSL                  true, if ssl connection
	 * @return the connection          
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetAddress address, int port, int connectTimeoutMillis, boolean isSSL) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(address, port), null, workerpool, connectTimeoutMillis, maxWaitMillis.get(), isSSL);
	}

	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 *  a new one will be created
	 *
	 * @param address     the server address
	 * @param port        the server port
 	 * @param appHandler  the application handler (supported: IConnectHandler, IDisconnectHandler, IDataHandler, IIdleTimeoutHandler and IConnectionTimeoutHandler)
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetAddress address, int port, IHandler appHandler) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(address, port), appHandler, workerpool, Integer.MAX_VALUE, maxWaitMillis.get(), false);
	}



	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 *  a new one will be created
	 *
	 * @param address     the server address
	 * @param port        the server port
 	 * @param appHandler  the application handler (supported: IConnectHandler, IDisconnectHandler, IDataHandler, IIdleTimeoutHandler and IConnectionTimeoutHandler)
	 * @param isSSL       true, if ssl connection
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetAddress address, int port, IHandler appHandler, boolean isSSL) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(address, port), appHandler, workerpool, Integer.MAX_VALUE, maxWaitMillis.get(), isSSL);
	}



	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 *  a new one will be created
	 *
	 * @param host        the server address
	 * @param port        the server port
 	 * @param appHandler  the application handler (supported: IConnectHandler, IDisconnectHandler, IDataHandler, IIdleTimeoutHandler and IConnectionTimeoutHandler)
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(String host, int port, IHandler appHandler) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(host, port), appHandler, workerpool, Integer.MAX_VALUE, maxWaitMillis.get(), false);
	}


	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 *  a new one will be created
	 *
	 * @param host        the server address
	 * @param port        the server port
 	 * @param appHandler  the application handler (supported: IConnectHandler, IDisconnectHandler, IDataHandler, IIdleTimeoutHandler and IConnectionTimeoutHandler)
	 * @param isSSL       true, if ssl connection
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(String host, int port, IHandler appHandler, boolean isSSL) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(host, port), appHandler, workerpool, Integer.MAX_VALUE, maxWaitMillis.get(), isSSL);
	}


	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 *  a new one will be created
	 *
	 * @param address                the server address
	 * @param port                   the server port
 	 * @param appHandler             the application handler (supported: IConnectHandler, IDisconnectHandler, IDataHandler, IIdleTimeoutHandler and IConnectionTimeoutHandler)
	 * @param connectTimeoutMillis  the connection creation timeout
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetAddress address, int port, IHandler appHandler, int connectTimeoutMillis) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(address, port), appHandler, workerpool, connectTimeoutMillis, maxWaitMillis.get(), false);
	}


	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 *  a new one will be created
	 *
	 * @param address                the server address
	 * @param port                   the server port
 	 * @param appHandler             the application handler (supported: IConnectHandler, IDisconnectHandler, IDataHandler, IIdleTimeoutHandler and IConnectionTimeoutHandler)
	 * @param connectTimeoutMillis  the connection creation timeout
	 * @param isSSL                 true, if ssl connection
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetAddress address, int port, IHandler appHandler, int connectTimeoutMillis, boolean isSSL) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(address, port), appHandler, workerpool, connectTimeoutMillis, maxWaitMillis.get(), isSSL);
	}



	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created <br> <br>
	 *
	 * This method is thread safe
	 *
	 * @param address               the server address
	 * @param port                  the sever port
 	 * @param appHandler            the application handler (supported: IConnectHandler, IDisconnectHandler, IDataHandler, IIdleTimeoutHandler and ConnectionTimeoutHandler)
	 * @param workerPool            the worker pool to use
	 * @param connectTimeoutMillis  the connection creation timeout
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetAddress address, int port, IHandler appHandler, Executor workerPool, int connectTimeoutMillis) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(address, port), appHandler, workerPool, connectTimeoutMillis, maxWaitMillis.get(), false);
	}


	/**
	 * get a pool connection for the given address. If no free connection is in the pool,
	 * a new one will be created <br> <br>
	 *
	 * This method is thread safe
	 *
	 * @param address               the server address
	 * @param port                  the sever port
 	 * @param appHandler            the application handler (supported: IConnectHandler, IDisconnectHandler, IDataHandler, IIdleTimeoutHandler and ConnectionTimeoutHandler)
	 * @param workerPool            the worker pool to use
	 * @param connectTimeoutMillis  the connection creation timeout
	 * @param isSSL                 true, if ssl connection
	 * @return the connection
	 * @throws SocketTimeoutException if the wait timeout has been reached (this will only been thrown if wait time has been set)
	 * @throws IOException  if an exception occurs
	 */
	public INonBlockingConnection getNonBlockingConnection(InetAddress address, int port, IHandler appHandler, Executor workerPool, int connectTimeoutMillis, boolean isSSL) throws IOException, SocketTimeoutException {
		return getConnection(new InetSocketAddress(address, port), appHandler, workerPool, connectTimeoutMillis, maxWaitMillis.get(), isSSL);
	}

	
	
	
	private INonBlockingConnection getConnection(InetSocketAddress address, IHandler appHandler, Executor workerPool, int connectTimeoutMillis, long waitTimeMillis, boolean isSSL) throws IOException,  SocketTimeoutException {
		
		if (isOpen.get()) {
			
			// try to get a pooled native connection
			NativeConnectionHolder nativeConnectionHolder = pool.getAndRemoveIdleConnection(address, isSSL);

			// got it?
			if (nativeConnectionHolder != null) {

				// reset resource (connection will be closed if invalid)
				boolean isValid = nativeConnectionHolder.isVaild(System.currentTimeMillis(), true);
				if (isValid && nativeConnectionHolder.getConnection().reset()) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("got connection from pool (" + pool.toString() + ", idleTimeoutMillis=" + getPooledMaxIdleTimeMillis() + ", lifeTimeout=" + getPooledMaxLifeTimeMillis() + "): " + nativeConnectionHolder.getConnection());
					}
					return new NonBlockingConnectionProxy(nativeConnectionHolder, appHandler);
					
				// resource is not valid -> call recursive
				} else {
					return getConnection(address, appHandler, workerPool, connectTimeoutMillis, waitTimeMillis, isSSL);
				}

				
			// ... no resource available
			} else {
				
				// active size smaller than max size? -> create a new one
				if (pool.getNumActive() < maxActive.get()) {
					nativeConnectionHolder = newNativeConnection(address, workerPool, connectTimeoutMillis, isSSL);
					return new NonBlockingConnectionProxy(nativeConnectionHolder, appHandler);

				
				// max size reached wait for free a resource
				} else {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("no free resources available waiting max " + DataConverter.toFormatedDuration(maxWaitMillis.get()) + " for a free resource (" + pool.toString() + ", idleTimeoutMillis=" + getPooledMaxIdleTimeMillis() + ", lifeTimeout=" + getPooledMaxLifeTimeMillis() + ")");
					}
					
					long startTime = System.currentTimeMillis();
  
					synchronized (retrieveGuard) {
						try {
							retrieveGuard.wait(waitTimeMillis);
						} catch (InterruptedException ignore) { }
					}

					int elapsed = (int) (System.currentTimeMillis() - startTime);
					long remaining = waitTimeMillis - elapsed;
					if (remaining > 0) {
						return getConnection(address, appHandler, workerPool, connectTimeoutMillis, remaining, isSSL);
						
					} else {
						if (LOG.isLoggable(Level.FINE)) {
							LOG.fine("wait timeout " + maxWaitMillis.get() + " reached");
						}
						throw new SocketTimeoutException("wait timeout reached"); 
					}
				}
			}

		} else  {
			throw new IOException("pool is already closed");
		}		
	}

	
	private void returnToIdlePool(NativeConnectionHolder nativeConnectionHolder) {
		pool.returnIdleConnection(nativeConnectionHolder);

		if (maxActive.get() < Integer.MAX_VALUE) {
			wakeupPendingRetrieve();
		}
	}
	
	
	private void wakeupPendingRetrieve() {
		synchronized (retrieveGuard) {
			retrieveGuard.notifyAll();
		}
	}
	

	private NativeConnectionHolder newNativeConnection(InetSocketAddress address, Executor workerPool, int creationTimeoutMillis, boolean isSSL) throws IOException {

		int trials = 0;
		int sleepTime = 3;
		long start = System.currentTimeMillis();

		
		while (true) {
	
			try {
				
				// create a new tcp connection and the assigned holder
				NativeConnectionHolder nativeConnectionHolder = new NativeConnectionHolder();
				NonBlockingConnection pooledConnection = new NonBlockingConnection(address, true, creationTimeoutMillis, new HashMap<String, Object>(), sslContext, isSSL, nativeConnectionHolder, workerPool);
				nativeConnectionHolder.init(pooledConnection);
				
				return nativeConnectionHolder;

			} catch (IOException ioe) {
				
				// timeout reached
				if (System.currentTimeMillis() > (start + IConnectionPool.DEFAULT_CREATION_TIMEOUT_MILLIS)) {
					countCreationError.incrementAndGet();

					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("error occured by creating connection to " + address
							   + ". creation timeout " + IConnectionPool.DEFAULT_CREATION_TIMEOUT_MILLIS + " reached.");
					}

					throw new IOException("could not connect to " + address + " (" + trials + " trials) " + ioe.toString());

				// .. no increase sleep time, wait and retry
				} else {
					sleepTime = sleepTime * 3;
					
					try {
						Thread.sleep(sleepTime);
					} catch (InterruptedException ignore) { }
				}
			}
		} 
	}

	
	/**
	 * {@inheritDoc}
	 */
	public boolean isOpen() {
		return isOpen.get();
	}

	
	/**
	 * {@inheritDoc}
	 */
	public void close() {

		if (isOpen.getAndSet(false)) {
			pool.close();
			
			stopWatchdog();

			for (ILifeCycle lifeCycle : listeners) {
				try {
					lifeCycle.onDestroy();
				} catch (IOException ioe) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("exception occured by destroying " + lifeCycle + " " + ioe.toString());
					}
				}
			}
			listeners.clear();
						
			// unset references
			workerpool = null;
		}
	}
	
	

	/**
	 * destroy the pool by killing idle and active connections
	 */
	public void destroy() {

		if (isOpen.getAndSet(false)) {
			pool.destroy();
			
			stopWatchdog();
	
			for (ILifeCycle lifeCycle : listeners) {
				try {
					lifeCycle.onDestroy();
				} catch (IOException ioe) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("exception occured by destroying " + lifeCycle + " " + ioe.toString());
					}
				}
			}
			listeners.clear();
						
			// unset references
			workerpool = null;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void addListener(ILifeCycle listener) {
		listeners.add(listener);
	}


	/**
	 * {@inheritDoc}
	 */
	public boolean removeListener(ILifeCycle listener) {
		boolean result = listeners.remove(listener);

		return result;
	}

	
	/**
	 * set the worker pool which will be assigned to the connections for call back handling
	 * @param workerpool the worker pool
	 */
	public void setWorkerpool(Executor workerpool) {
		this.workerpool = workerpool;
	}
	
	
	/**
	 * get the worker pool which will be assigned to the connections for call back handling
	 * @return the worker pool
	 */
	public Executor getWorkerpool() {
		return workerpool;
	}
	


	/**
	 * {@inheritDoc}
	 */
	public int getMaxActive() {
		return maxActive.get();
    }


	/**
	 * {@inheritDoc}
	 */
    public void setMaxActive(int maxActive) {
    	this.maxActive.set(maxActive);
    	wakeupPendingRetrieve(); 
    }

  
    
    /**
	 * {@inheritDoc}
	 */
    public long getCreationMaxWaitMillis() {
        return maxWaitMillis.get();
    }


    /**
	 * {@inheritDoc}
	 */
    public void setCreationMaxWaitMillis(long maxWaitMillis) {
    	this.maxWaitMillis.set(maxWaitMillis);
    }


    /**
	 * {@inheritDoc}
	 */
    public int getMaxIdle() {
    	return maxIdle.get();
    }


    /**
	 * {@inheritDoc}
	 */
    public void setMaxIdle(int maxIdle) {
    	this.maxIdle.set(maxIdle);
    }


    /**
	 * {@inheritDoc}
	 */
    public int getNumActive() {
        return pool.getNumActive();
    }

 
    public List<String> getActiveConnectionInfos() {
    	List<String> result = new ArrayList<String>();
	    	
    	List<NativeConnectionHolder> connectionHolders = pool.newManagedPoolCopy();
    	connectionHolders.removeAll(pool.newIdleCopySet());
    	
    	for (NativeConnectionHolder connectionHolder : connectionHolders) {
    		result.add(connectionHolder.toString());
    	}
	        
    	return result;
    }
    
    
    public List<String> getIdleConnectionInfos() {
    	List<String> result = new ArrayList<String>();

    	for (NativeConnectionHolder nativeConnectionHolder : pool.newIdleCopySet()) {
    		result.add(nativeConnectionHolder.toString());
    	}
	        
    	return result;
    }
    

    /**
	 * {@inheritDoc}
	 */
    public int getNumIdle() {
    	return pool.getNumIdle();
	}

    /**
	 * {@inheritDoc}
	 */
    public int getNumCreated() {
    	return countCreated.get();
    }
    
    
    /**
	 * get the number of the creation errors 
	 * 
	 * @return the number of creation errors 
	 */
    public int getNumCreationError() {
    	return countCreationError.get();
    }
    
    /**
	 * {@inheritDoc}
	 */
    public int getNumDestroyed() {
    	return countDestroyed.get();
    }

    
    int getNumIdleTimeout() {
    	return countIdleTimeout.get();
    }
    
    int getNumConnectionTimeout() {
    	return countConnectionTimeout.get(); 
    }
  
    
    int getNumPoolIdleTimeout() {
    	return countTimeoutPooledIdle.get();
    }

    int getNumPoolLifetimeTimeout() {
    	return countTimeoutPooledLifetime.get();
    }
    
    /**
	 * {@inheritDoc}
	 */
    public int getNumTimeoutPooledMaxIdleTime() {
    	return countTimeoutPooledIdle.get();
    }

    /**
	 * {@inheritDoc}
	 */
    public int getNumTimeoutPooledMaxLifeTime() {
    	return countTimeoutPooledLifetime.get();
    }
    
    
    /**
	 * {@inheritDoc}
	 */
	public int getPooledMaxIdleTimeMillis() {
		return poolIdleTimeoutMillis.get();
	}


	/**
	 * {@inheritDoc}
	 */
	public void setPooledMaxIdleTimeMillis(int idleTimeoutMillis) {
		this.poolIdleTimeoutMillis.set(idleTimeoutMillis);
		updateWatchdog(idleTimeoutMillis);
	}


	/**
	 * {@inheritDoc}
	 */
	public int getPooledMaxLifeTimeMillis() {
		return lifeTimeoutMillis.get();
	}


	/**
	 * {@inheritDoc}
	 */
	public void setPooledMaxLifeTimeMillis(int lifeTimeoutMillis) {
		this.lifeTimeoutMillis.set(lifeTimeoutMillis);
		updateWatchdog(lifeTimeoutMillis);
	}

	
	private synchronized void updateWatchdog(int requiredTimeMillis) {
		
		if (isOpen.get()) {
			
			if (requiredTimeMillis < (watchdogCheckPeriod * 5)) {
				if (watchdog != null) {
					watchdog.cancel();
				}
				
				watchdogCheckPeriod =  requiredTimeMillis / 5;
				if (watchdogCheckPeriod < 500) {
					watchdogCheckPeriod = 500;
				}
				
				watchdog = new Watchog();
				IoProvider.getTimer().schedule(watchdog, watchdogCheckPeriod, watchdogCheckPeriod);
			}
		}
	}

	

	private synchronized void stopWatchdog() {
		if (watchdog != null) {
			watchdog.cancel();
		}
	}


	/**
	 * get the current number of pending get operations to retrieve a resource
	 *
	 * @return the current number of pending get operations
	 */
	public int getNumPendingGet() {
		return countPendingGet.get();
	}

	


	/**
	 * {@inheritDoc}
	 */
	public static void destroy(INonBlockingConnection connection) throws IOException {
		if (connection == null) {
			if (LOG.isLoggable(Level.FINE)) {
				LOG.fine("warning trying to destroy a <null> connection. destroy will be ignored");
			}
			return;
		}

		if (connection instanceof NonBlockingConnectionProxy) {
			((NonBlockingConnectionProxy) connection).destroy();

		} else {
			connection.close();
		}
	}
	
	
	static boolean isDestroyed(INonBlockingConnection connection) {
		if (connection instanceof NonBlockingConnectionProxy) {
			return ((NonBlockingConnectionProxy) connection).isDestroyed();
		
		} else {
			return connection.isOpen();
		}
	}
	
	
	private void checkIdleConnections() {
		long currentMillis = System.currentTimeMillis();
		
		for (NativeConnectionHolder nativeConnectionHolder : pool.newIdleCopySet()) {
			if (!nativeConnectionHolder.isVaild(currentMillis, false)) {
				boolean isRemoved = pool.removeIdleConnection(nativeConnectionHolder);
				if (isRemoved) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("[" + nativeConnectionHolder.getId() + "] closing connection because it is invalid");
					}
					nativeConnectionHolder.isVaild(currentMillis, true);  // true -> closes the connection 
					nativeConnectionHolder.close();  /// sanity close
				}
			}
		}		
	}
	   
	
	
	@Override
	public String toString() {
		
		StringBuilder sb = new StringBuilder();
		sb.append("active=" + getNumActive() + ", idle=" + getNumIdle() + " "); 
		sb.append("created=" + countCreated + ", destroyed=" + countDestroyed + " (");
		sb.append("maxActive=" + maxActive + ", maxIdle=" + maxIdle + ", maxWaitTimeMillis=" + maxWaitMillis + ", ");
		sb.append("poolIdleTimeout=" + poolIdleTimeoutMillis + ", poollifetimeTimeout=" + lifeTimeoutMillis + ")");
		
		return sb.toString();
	}
	
	
	private final class Watchog extends TimerTask {
		
		@Override
		public void run() {
			try {
				checkIdleConnections();
			} catch (Throwable t) {
				if (LOG.isLoggable(Level.FINE)) {
					LOG.fine("error occured by checking connections " + t.toString());
				}
			}
		}
	}
	


	private static final class Pool {
		
		private final ArrayList<NativeConnectionHolder> managedPool = new ArrayList<NativeConnectionHolder>();
		private final HashMap<InetSocketAddress, List<NativeConnectionHolder>> idlePool = new HashMap<InetSocketAddress, List<NativeConnectionHolder>>();
		private int numIdle = 0;

		private boolean isOpen = true;
		
		
		public void register(NativeConnectionHolder nativeConnectionHolder) {
			synchronized (this) {
				if (isOpen) {
					managedPool.add(nativeConnectionHolder);
				} else {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("ignore registering connection " + nativeConnectionHolder.toString() +  " because pool is already closed");
					}
				}
			}
			
			if (LOG.isLoggable(Level.FINE)) {
				LOG.fine("[" + nativeConnectionHolder.getId() + "] added to managed pool (active=" + getNumActive() + ", idle=" + getNumIdle() + ")");
			}
		}
		

	    public boolean remove(NativeConnectionHolder nativeConnectionHolder) {
	    	boolean isRemoved = false;
	    	
	    	synchronized (this) {
		    	isRemoved = managedPool.remove(nativeConnectionHolder);
	    	}
	    	
			if (LOG.isLoggable(Level.FINE)) {
				if (isRemoved) {
					LOG.fine("[" + nativeConnectionHolder.getId() + "] connection removed from managed pool (active=" + getNumActive() + ", idle=" + getNumIdle() + ")");
				} else {
					LOG.fine("[" + nativeConnectionHolder.getId() + "] could not removed connection from managed pool. Connection already removed? (active=" + getNumActive() + ", idle=" + getNumIdle() + ")");					
				}
			}
			
			return isRemoved;
	    }

	    
	    public boolean removeIdleConnection(NativeConnectionHolder connectionHolder) {
	    	
	    	synchronized (this) {
	    		List<NativeConnectionHolder> idleList = idlePool.get(connectionHolder.getAddress());
		    	if (idleList != null) {

					for (NativeConnectionHolder nativeConnectionHolder : idleList) {
						if (nativeConnectionHolder == connectionHolder) {
							boolean isRemoved = idleList.remove(nativeConnectionHolder);
							if (idleList.isEmpty()) {
								idlePool.remove(connectionHolder.getAddress());
							}
							
							if (isRemoved) {
								numIdle--;
								assert (numIdle == computeNumIdle()) : "numIdle delta occured";
							}
							
							return isRemoved;
						}
					}
				}
			}
	    	
	    	return false;
	    }

	    
	    
	    
	    
	    public NativeConnectionHolder getAndRemoveIdleConnection(InetSocketAddress address, boolean isSSL) {
	    	synchronized (this) {
	    		List<NativeConnectionHolder> idleList = idlePool.get(address);
		    	if (idleList != null) {

					for (NativeConnectionHolder nativeConnectionHolder : idleList) {
						if (nativeConnectionHolder.isSSL == isSSL) {
							idleList.remove(nativeConnectionHolder);
							if (idleList.isEmpty()) {
								idlePool.remove(address);
							}

							if (LOG.isLoggable(Level.FINE)) {
								LOG.fine("[" + nativeConnectionHolder.getId() + "] got from idle pool (active=" + getNumActive() + ", idle=" + getNumIdle() + ")");
							}
							
							numIdle--;
							assert (numIdle == computeNumIdle()) : "numIdle delta occured";
							
							return nativeConnectionHolder;
						}
					}
				}
			}
	    	
	    	return null;
	    }

	    
	    
	    public void returnIdleConnection(NativeConnectionHolder nativeConnectionHolder) {
	  		InetSocketAddress address = nativeConnectionHolder.getAddress();
	    	
	    	synchronized (this) {
	    		List<NativeConnectionHolder> idleList = idlePool.get(address);
		    	if (idleList == null) {
		    		idleList = new ArrayList<NativeConnectionHolder>();
		    		idlePool.put(address, idleList);
		    	}
		    	
		    	idleList.add(nativeConnectionHolder);

		    	numIdle++;
				assert (numIdle == computeNumIdle()) : "numIdle delta";
			}
	    	
			if (LOG.isLoggable(Level.FINE)) {
				LOG.fine("[" + nativeConnectionHolder.getId() + "] added to idle pool (active=" + getNumActive() + ", idle=" + getNumIdle() + ")");
			}
	    }

	    
    
	    
	    public void close() {	    	
	    	synchronized (this) {
	    		if (isOpen) {
	    			isOpen = false;
	    		} else {
	    			return;
	    		}
	    	}

	
	    	List<NativeConnectionHolder> idleSet = newIdleCopySet();
	    	
			if (LOG.isLoggable(Level.FINE)) {
				LOG.fine("closing " + idleSet.size() + " idle conection(s); " + managedPool.size() + " connection(s) stay open unmanaged");
			}
	    	
	    	for (NativeConnectionHolder nativeConnectionHolder : idleSet) {
				nativeConnectionHolder.close();
			}
	    	
	    	idlePool.clear();
	    	managedPool.clear();
	    	numIdle = 0;
		}

	    
	    public void destroy() {	    	
	    	synchronized (this) {
	    		if (isOpen) {
	    			isOpen = false;
	    		} else {
	    			return;
	    		}
	    	}

	
	    	List<NativeConnectionHolder> connections = newManagedPoolCopy();
	    	
			if (LOG.isLoggable(Level.FINE)) {
				LOG.fine("closing " + connections.size() + " managed connections");
			}
	    	
	    	for (NativeConnectionHolder nativeConnectionHolder : connections) {
				nativeConnectionHolder.close();
			}
	    	
	    	synchronized (this) {
		    	idlePool.clear();
		    	managedPool.clear();
		    	numIdle = 0;
	    	}
		}

	    
		private List<NativeConnectionHolder> newIdleCopySet() {
			List<NativeConnectionHolder> idleList = new ArrayList<NativeConnectionHolder>();
			
			for (List<NativeConnectionHolder> nativeConnectionHolderList : newIdlePoolCopy().values()) {
				idleList.addAll(nativeConnectionHolderList);
			}
			
			return idleList;
		}
		
		
		@SuppressWarnings("unchecked")
		List<NativeConnectionHolder> newManagedPoolCopy() {
			List<NativeConnectionHolder> managedPoolCopy = null;
			
			synchronized (this) {
				managedPoolCopy = (List<NativeConnectionHolder>) managedPool.clone();
			}
			
			return managedPoolCopy;
		}
		
	    
		@SuppressWarnings("unchecked")
		HashMap<InetSocketAddress, List<NativeConnectionHolder>> newIdlePoolCopy() {
			HashMap<InetSocketAddress, List<NativeConnectionHolder>> idlePoolCopy = null;
			
			synchronized (this) {
				idlePoolCopy = (HashMap<InetSocketAddress, List<NativeConnectionHolder>>) idlePool.clone();
			}
			
			return idlePoolCopy;
		}
		

	    
		public int getSize() {
			synchronized (this) {
				return managedPool.size();
			}
		}


		public int getNumIdle() {
			synchronized (this) {
				return numIdle;
			}
		}
		
		public int getNumActive() {
			synchronized (this) {
				return (managedPool.size() - numIdle);
			}
		}

		
		private int computeNumIdle() {
			int size = 0;
			
			for (List<NativeConnectionHolder> nativeConnectionHolderList : idlePool.values()) {
				size += nativeConnectionHolderList.size();
			}
			
			return size;
		}


		
	    
	    @Override
	    public String toString() {
	    	return "size=" + getSize() + ", active=" + getNumActive();
	    }
	}
	
	

	
	

	@Execution(Execution.NONTHREADED)
	private final class NativeConnectionHolder implements IDataHandler, IDisconnectHandler, IConnectionTimeoutHandler, IIdleTimeoutHandler {

		private final AtomicBoolean isClosed = new AtomicBoolean(false);
		private final AtomicBoolean isReusable = new AtomicBoolean(true);
		private final AtomicReference<NonBlockingConnectionProxy> connectionProxy = new AtomicReference<NonBlockingConnectionProxy>(null);

		private NonBlockingConnection connection;
		private InetSocketAddress address;
		private boolean isSSL;
		

		private int usage = 0;
		private long creationTimeMillis = System.currentTimeMillis();
		private long lastUsageTimeMillis = System.currentTimeMillis();

		
		
		public void init(NonBlockingConnection connection) {
			this.connection = connection;

			isSSL = connection.isSecure();
			address = new InetSocketAddress(connection.getRemoteAddress(), connection.getRemotePort());

			pool.register(this);
			countCreated.incrementAndGet();

			if (LOG.isLoggable(Level.FINE)) {
				LOG.fine("pooled connection created (" + pool.toString() + ", pooledIdleTimeoutMillis=" + getPooledMaxIdleTimeMillis() + ", pooledLifeTimeout=" + getPooledMaxLifeTimeMillis() + "): " + connection);
			}
		}
		

		String getId() {
			return connection.getId();
		}
		
		
		public boolean onData(INonBlockingConnection connection) throws IOException, BufferUnderflowException, MaxReadSizeExceededException {		
			NonBlockingConnectionProxy cp = connectionProxy.get();
			if (cp != null) {
				try {
					return cp.onData();
				} catch (MaxReadSizeExceededException mre) {
					isReusable.set(false);
					throw mre;
				} catch (IOException ioe) {
					isReusable.set(false);
					throw ioe;
				}
			} else {
				return true;
			}
		}
		
		
		
		public boolean onDisconnect(INonBlockingConnection connection) throws IOException {
			isReusable.set(false);

			try {
				NonBlockingConnectionProxy cp = connectionProxy.get();
				if (cp!= null) {
					return cp.onDisconnect();					
				} else {
					return true;
				}
				
			} finally {
				countDestroyed.incrementAndGet();
				pool.remove(this);
				
			    if (LOG.isLoggable(Level.FINE)) {
			    	LOG.fine("pooled connection (" + address.toString() + ") destroyed (" + pool.toString() + ", idleTimeoutMillis=" + getPooledMaxIdleTimeMillis() + ", lifeTimeout=" + getPooledMaxLifeTimeMillis() + "): " + connection);
			    }
			}
		}
		

		
		public boolean onConnectionTimeout(INonBlockingConnection connection) throws IOException {
		    isReusable.set(false);

		    countConnectionTimeout.incrementAndGet();
		    
		    if (LOG.isLoggable(Level.FINE)) {
		    	LOG.fine("[" + getId() + "] connection timeout occured");
		    }

		    
		    NonBlockingConnectionProxy cp = connectionProxy.get();
			if (cp != null) {
				return cp.onConnectionTimeout();
			} else {
				return false;  // let the connection be destroyed
			}
		}
		
		
		public boolean onIdleTimeout(INonBlockingConnection connection) throws IOException {
		    isReusable.set(false);

		    countIdleTimeout.incrementAndGet();

		    if (LOG.isLoggable(Level.FINE)) {
		    	LOG.fine("[" + getId() + "] idle timeout (" + DataConverter.toFormatedDuration(getConnection().getIdleTimeoutMillis()) + ") occured");
		    }
		    
		    NonBlockingConnectionProxy cp = connectionProxy.get();
			if (cp != null) {
				return cp.onIdleTimeout();
			} else {
				return false;  // let the connection be destroyed
			}
		}
		
		
		
		int getUsage() {
			return usage;
		}
		
		
		long getCreationTimeMillis() {
			return creationTimeMillis;
		}

		long getLastUsageTimeMillis() {
			return lastUsageTimeMillis;
		}

		void setReusable(boolean reusable) {
			isReusable.set(reusable);
		}

		NonBlockingConnection getConnection() {
			return connection;
		}

		InetSocketAddress getAddress() {
			return address;
		}

		
		boolean isSecure() {
			return isSSL;
		}


		
		void lease(NonBlockingConnectionProxy proxy) {
			usage++;
			lastUsageTimeMillis = System.currentTimeMillis();
			
			connectionProxy.set(proxy);
			if (proxy != null) {
				proxy.onConnect();
			}
		}

				
		void release() {
			NonBlockingConnectionProxy cp = connectionProxy.get();
			if (cp != null) {
		        if (LOG.isLoggable(Level.FINE)) {
		            LOG.fine("[" + getId() + "] connection will be released. calling onDisconnect");
		        }
                cp.onDisconnect();
            }
		    
            try {
                connection.getTaskQueue().performNonThreaded(new ReleaseTask(), workerpool);
            } catch (Exception e) {
                if (LOG.isLoggable(Level.FINE)) {
                    LOG.fine("[" + getId() + "] could not perform destroy task for " + connection.getId() + " Reason " + e.toString());
                }
                close();
            }
		}
		
		
		private final class ReleaseTask implements Runnable {
		    
		    public void run() {
                performRelease();
            }
		}

		
		   
		private void performRelease() {
		    
			try {
				lastUsageTimeMillis = System.currentTimeMillis();

				// is connection is open?
				if (!connection.isConnected() || !connection.isOpen() || !isOpen()) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("[" + getId() + "] do not return pooled connection (" + address.toString() + ") into to pool, because the connection or pool is closed");
					}
					
					close();
					return;
				}
				

				// reset resource
				boolean isValid = isVaild(System.currentTimeMillis(), true);
				if (isValid) {
	                // .. and return it to the pool if max idle size is not reached
					if ((maxIdle.get() != Integer.MAX_VALUE) || (pool.getNumIdle() >= maxIdle.get())) {
						return;
					}
					
					
					boolean isReset = connection.reset();
					if (isReset) {
						NonBlockingConnectionProxy cp = connectionProxy.get();
						if (cp != null) {
							cp.dettach();
						}
						
						if (LOG.isLoggable(Level.FINE)) {
							LOG.fine("[" + connection.getId() + "] releasing connection (for reuse)");
						}
						returnToIdlePool(this);
						
					} else {
						close();
					}
				} 

			} catch (Exception e) {
                // eat and log exception
				if (LOG.isLoggable(Level.FINE)) {
					LOG.fine("error occured by releasing a pooled connection (" + address.toString() + ") " + e.toString());
				}
			}
		}
		
	

		/**
		 * check if the connection is valid
		 * 
		 * @return true, if reuseable
		 */
		private boolean isVaild(long currentTimeMillis, boolean closeIfInvalid) {

			// is open?
			if (isClosed.get()) {
				return false;
			}
			
			
			if (!isReusable.get()) {
				if (closeIfInvalid) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("[" + getId() + "] closing connection because it is marked as non reuseable");
					}
					close();
				}
				return false;
			}

			if (!connection.isConnected() || !connection.isOpen()) {
				if (closeIfInvalid) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("[" + getId() + "] closing connection because it is disconnected or closed");
					}
					close();
				}
				return false;
			}
	
			
			if (connection.getRemainingMillisToIdleTimeout() < MIN_REMAINING_MILLIS_TO_IDLE_TIMEOUT) {
					if (closeIfInvalid) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("[" + getId() + "] closing connection because remaining time to idle timeout (" + connection.getRemainingMillisToIdleTimeout() + " millis) is to small");
					}
					countRemainingMillisToIdleTimeoutToSmall.incrementAndGet();
					close();
					}
				return false;
			}
	
				
			if (connection.getRemainingMillisToConnectionTimeout() < MIN_REMAINING_MILLIS_TO_CONNECTION_TIMEOUT) {
				if (closeIfInvalid) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("[" + getId() + "] closing connection because remaining time to connection timeout (" + connection.getRemainingMillisToConnectionTimeout() + " millis)  is to small");
					}
					countRemainingConnectionToIdleTimeoutToSmall.incrementAndGet();
					close();
				}
				return false;
			}


			if ((poolIdleTimeoutMillis.get() != Integer.MAX_VALUE) && (currentTimeMillis > (lastUsageTimeMillis + poolIdleTimeoutMillis.get()))) {
				if (closeIfInvalid) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("[" + connection.getId() + "] connection (" + address + ") pool idle timeout reached (" + poolIdleTimeoutMillis + ")"); 
					}
					countTimeoutPooledIdle.incrementAndGet();
					close();
				}
				return false;
			}


			
			
			if ((lifeTimeoutMillis.get() != Integer.MAX_VALUE) && (currentTimeMillis > (creationTimeMillis + lifeTimeoutMillis.get()))) {
				if (closeIfInvalid) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("[" + getId() + "] connection (" + address + ") pool life timeout reached (" + lifeTimeoutMillis + ")"); 
					}
					countTimeoutPooledLifetime.incrementAndGet();
					close();
				}
				return false;
			}
	
			return true;
		}

		
		

		void close() {
			if (!isClosed.getAndSet(true)) {
				NonBlockingConnectionProxy cp = connectionProxy.get();
			    if (cp != null) {
			        cp.onDisconnect();
			    }
		            
			    pool.remove(this);
			    wakeupPendingRetrieve();
	
			    try {
			    	connection.close();
			    } catch (Exception e) {
			    	// eat and log exception
			    	if (LOG.isLoggable(Level.FINE)) {
			    		LOG.fine("[" + getId() + "] error occured by closing connection (" + address.toString() + ") " + connection + ": " + e.toString());
			    	}
			    }
			}
		}		    

		

		
		@Override
		public String toString() {
			StringBuilder sb = new StringBuilder();

			if (connection.isReceivingSuspended()) {
				sb.append("[suspended] ");
			}
			
			sb.append(connection.getLocalAddress() + ":" + connection.getLocalPort() + " -> " +
		   			  connection.getRemoteAddress() + ":" + connection.getRemotePort() + " " +
		   			  "[" + connection.getId() + "]");
			
		
			SimpleDateFormat df = new SimpleDateFormat("yyyy.MM.dd HH:mm:ss");

			sb.append(" creationTime=" + df.format(getCreationTimeMillis()) + 
					  ", lastUsageTime=" + df.format(getLastUsageTimeMillis()) +
					  ", countUsage=" + getUsage() +
					  ", isReusable=" + isReusable.get());

			return sb.toString();
		}
	}

	
	
	


	private static final class NonBlockingConnectionProxy implements INonBlockingConnection {

		private volatile boolean isOpen = true;
		
		private volatile boolean isDetached = false;
		private NativeConnectionHolder nativeConnectionHolder = null;
		private final String id;

		
		private HandlerAdapter handlerAdapter = null;

		private Object attachment = null;
		
		// timeout support
		private final AtomicBoolean idleTimeoutOccured = new AtomicBoolean(false);
		private final AtomicBoolean connectionTimeoutOccured = new AtomicBoolean(false);
		private final AtomicBoolean disconnectOccured = new AtomicBoolean(false);

		
		
		NonBlockingConnectionProxy(NativeConnectionHolder holder, IHandler appHandler) throws IOException {
			nativeConnectionHolder = holder;
			id = holder.getConnection().getId() + "I" + Integer.toHexString(holder.usage);
						
			handlerAdapter = HandlerAdapter.newInstance(appHandler);
			holder.lease(this); // onConnect will be called implicit
		}

		
		private void ensureOpen() {
			if (!isOpen) {
				throw new RuntimeException("channel " + getId() + " is closed");
			}
		}
		
		void dettach() {
			isDetached = true;
			nativeConnectionHolder = null;
		}

		public void setHandler(IHandler hdl) throws IOException {
			ensureOpen();
			
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				handlerAdapter = HandlerAdapter.newInstance(hdl);
				if (!holder.getConnection().isReadBufferEmpty()) {
					onData();
				}
			}
		}
		
		
		public IHandler getHandler() {
			ensureOpen();
			
			if (isDetached) {
				return null;
			}
			
			if (handlerAdapter == null) {
				return null;
			} else {
				return ((HandlerAdapter) handlerAdapter).getHandler();
			}
		}
		
		boolean isDestroyed() {
			if (isDetached) {
				return true;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return !holder.getConnection().isConnected();
			} else {
				return true;
			}
		}

		
		public boolean isOpen() {
			if (!isOpen) {
				return false;
			}
			
			if (isDetached) {
				return false;
				
			} else {
				NativeConnectionHolder holder = nativeConnectionHolder;
				if (holder != null) {
					return holder.getConnection().isOpen();
				} else {
					return false;
				}
			}
		}

		
		public void close() throws IOException {
			
			if (isOpen()) {
				NativeConnectionHolder holder = nativeConnectionHolder;
				if (holder != null) {
					holder.release();   // onDisconnect will be called implicit
				}
			}
		}


		void destroy() {
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				try {
					holder.close();
				} catch (Exception e) {
	                // eat and log exception
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("[" + getId() + "] error occured while destroying pooledConnectionHolder " + nativeConnectionHolder + " reason: " + e.toString());
					}
				}
			}
		}


		@Override
		public String toString() {
			StringBuilder sb = new StringBuilder();

			sb.append(getId());
			if (!isOpen()) {
				sb.append(" closed");
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				sb.append(" (" + holder.getAddress() + ") ");
			}

			sb.append(" (proxy ");
			if (holder == null) {
				sb.append("closed ");
			} else {	
				sb.append("countUsage=" + holder.getUsage() + ", dateLastUsage=" + DataConverter.toFormatedDate(holder.getLastUsageTimeMillis()));
			}

			return sb.toString();
		}
		

		public String getId() {
			return id;
		}
		
		
		private boolean onConnect() {
			if (isDetached) {
				return false;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				try {
					return handlerAdapter.onConnect(this, holder.connection.getTaskQueue());
				} catch (IOException ioe) {
					if (LOG.isLoggable(Level.FINE)) {
						LOG.fine("[" + getId() + "] Error occured by perform onConnect callback on " + handlerAdapter + " " + ioe.toString());
					}
					return false;
				}
			} else {
				return false;
			}
		}
		

		
		private boolean onDisconnect() {
			
			if (!disconnectOccured.getAndSet(true)) {
				if (isDetached) {
					return false;
				}
				
				NativeConnectionHolder holder = nativeConnectionHolder;
				if (holder != null) {
					try {
						return handlerAdapter.onDisconnect(this, holder.connection.getTaskQueue());
					} catch (IOException ioe) {
						if (LOG.isLoggable(Level.FINE)) {
							LOG.fine("[" + getId() + "] Error occured by perform onDisconnect callback on " + handlerAdapter + " " + ioe.toString());
						}
					}
				}
			} 
			
			return false;
		}
		
		
		private boolean onData() throws IOException, MaxReadSizeExceededException {
			if (isDetached) {
				return true;
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return handlerAdapter.onData(this, holder.connection.getTaskQueue(), false);
			} else {
				if (LOG.isLoggable(Level.FINE)) {
					LOG.fine("[" + getId() + "] onData called even though proxy is closed");
				}
				return true;
			}
		}

		
		private boolean onConnectionTimeout() throws IOException {
			if (isDetached) {
				return true;
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				if (!connectionTimeoutOccured.getAndSet(true)) {
					return handlerAdapter.onConnectionTimeout(this, holder.connection.getTaskQueue());
				} else {
					return true;
				}
			} else {
				if (LOG.isLoggable(Level.FINE)) {
					LOG.fine("[" + getId() + "] onConnectionTimeout called even though proxy is closed");
				}
				return true;
			}
		}

		
		private boolean onIdleTimeout() throws IOException {
			if (isDetached) {
				return true;
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				if (!idleTimeoutOccured.getAndSet(true)) {
					return handlerAdapter.onIdleTimeout(this, holder.connection.getTaskQueue());
				} else {
					return true;
				}
			} else {
				if (LOG.isLoggable(Level.FINE)) {
					LOG.fine("[" + getId() + "] onIdletimeout called even though proxy is closed");
				}
				return true;
			}
		}

		void setReusable(boolean isReusable) {
			if (isDetached) {
				return;
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.setReusable(isReusable);
			} 
		}

		
		public boolean isServerSide() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			} 
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().isServerSide();
			} else {
				throw newClosedChannelRuntimeException();
			}		
		}
		
		
		public Executor getWorkerpool() {
			if (isDetached) {
				return NonBlockingConnection.getDefaultWorkerpool();
			} 
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().getWorkerpool();
			} else {
				return NonBlockingConnection.getDefaultWorkerpool();
			}
		}

		public void setWorkerpool(Executor workerpool) {
			if (isDetached) {
				return;
			} 
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().setWorkerpool(workerpool);
			} 
		}


		public void setAttachment(Object obj) {
			this.attachment = obj;
		}

		
		public Object getAttachment() {
			return attachment;
		}

		
		public void setAutoflush(boolean autoflush) {
			if (isDetached) {
				return;
			} 

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().setAutoflush(autoflush);
			} 
		}

		public boolean isAutoflush() {
			if (isDetached) {
				return IConnection.DEFAULT_AUTOFLUSH;
			} 

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().isAutoflush();
			} else {
				return IConnection.DEFAULT_AUTOFLUSH;
			}
		}

		
		public void setEncoding(String defaultEncoding) {
			if (isDetached) {
				return;
			} 

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().setEncoding(defaultEncoding);
			} 			
		}

		public String getEncoding() {
			if (isDetached) {
				return IConnection.INITIAL_DEFAULT_ENCODING;
			} 

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getEncoding();
			} else {
				return IConnection.INITIAL_DEFAULT_ENCODING;
			}								
		}

		public void setFlushmode(FlushMode flushMode) {
			if (isDetached) {
				return;
			} 

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().setFlushmode(flushMode);
			} 										
		}

		public FlushMode getFlushmode() {
			if (isDetached) {
				return IConnection.DEFAULT_FLUSH_MODE;
			} 

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getFlushmode();
			} else {
				return IConnection.DEFAULT_FLUSH_MODE;
			}															
		}
		
		public void setOption(String name, Object value) throws IOException {
			if (isDetached) {
				return;
			} 

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().setOption(name, value);
			} 																		
		}

		public Object getOption(String name) throws IOException {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			} 

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getOption(name);
			} else {
				throw newClosedChannelRuntimeException();
			}																							
		}

		@SuppressWarnings("unchecked")
		public Map<String, Class> getOptions() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			} 
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getOptions();
			} else {
				throw newClosedChannelRuntimeException();
			}																											
		}

		public void setWriteTransferRate(int bytesPerSecond) throws ClosedChannelException, IOException {
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().setWriteTransferRate(bytesPerSecond);
			} 																													
		}
		
		public int getWriteTransferRate() throws ClosedChannelException, IOException {
			if (isDetached) {
				return INonBlockingConnection.UNLIMITED;
			} 
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getWriteTransferRate();
			} else {
				return INonBlockingConnection.UNLIMITED;
			}																																			
		}
		
		public int getMaxReadBufferThreshold() {
			if (isDetached) {
				return Integer.MAX_VALUE;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getMaxReadBufferThreshold();
			} else {
				return Integer.MAX_VALUE;
			}																																							
		}
		
		public void setMaxReadBufferThreshold(int size) {
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().setMaxReadBufferThreshold(size);
			} 																																						
		}

		public void setConnectionTimeoutMillis(long timeoutMillis) {
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().setConnectionTimeoutMillis(timeoutMillis);
			} 																																										
		}

		public long getConnectionTimeoutMillis() {
			if (isDetached) {
				return Long.MAX_VALUE;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getConnectionTimeoutMillis();
			} else {
				return Long.MAX_VALUE;
			}
		}

		public void setIdleTimeoutMillis(long timeoutInMillis) {
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().setIdleTimeoutMillis(timeoutInMillis);
				idleTimeoutOccured.set(false);
			} else {
				throw newClosedChannelRuntimeException();
			}																																															
		}

		public long getIdleTimeoutMillis() {
			if (isDetached) {
				return Long.MAX_VALUE;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getIdleTimeoutMillis();
			} else {
				return Long.MAX_VALUE;
			}																																											
		}

		public long getRemainingMillisToConnectionTimeout() {
			if (isDetached) {
				return Long.MAX_VALUE;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getConnectionTimeoutMillis();
			} else {
				return Long.MAX_VALUE;
			}																																							
		}

		public long getRemainingMillisToIdleTimeout() {
			if (isDetached) {
				return Long.MAX_VALUE;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getRemainingMillisToIdleTimeout();
			} else {
				return Long.MAX_VALUE;
			}																																											
		}

		public boolean isSecure() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			} 
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().isSecure();
			} else {
				throw newClosedChannelRuntimeException();
			}		
		}

		
		public void activateSecuredMode() throws IOException {
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().activateSecuredMode();
			} 																																										
		}
		
		public boolean isSecuredModeActivateable() {
			if (isDetached) {
				return false;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().isSecuredModeActivateable();
			} else {
				return false;
			}																																															
		}

		
		public void suspendReceiving() throws IOException {
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().suspendRead();
			} 																																														
		}

		public void suspendRead() throws IOException {
			suspendReceiving();
		}
		
		
		public boolean isReceivingSuspended() {
			if (isDetached) {
				return false;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().isReadSuspended();
			} else {
				return false;
			}																																															
		}
		
		public boolean isReadSuspended() {
			return isReceivingSuspended();
		}

		
		public void resumeReceiving() throws IOException {
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().resumeRead();
			} 																																																	
		}
		
		
		public void resumeRead() throws IOException {
			resumeReceiving();
		}

		public InetAddress getLocalAddress() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().getLocalAddress();
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}

		public int getLocalPort() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().getLocalPort();
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}

		public InetAddress getRemoteAddress() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().getRemoteAddress();
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}

		public int getRemotePort()  {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().getRemotePort();
			} else {
				throw newClosedChannelRuntimeException();
			}																																																							
		}


		public int write(byte b) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(b);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public int write(byte... bytes) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(bytes);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		
		public void write(byte[] bytes, IWriteCompletionHandler writeCompletionHandler) throws IOException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().write(bytes, writeCompletionHandler);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		
		public int write(byte[] bytes, int offset, int length) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(bytes, offset, length);
			} else {
				throw newClosedChannelException();
			}																																															
		}
		
		
		public void write(byte[] bytes, int offset, int length, IWriteCompletionHandler writeCompletionHandler) throws IOException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().write(bytes, offset, length, writeCompletionHandler);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public int write(ByteBuffer buffer) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(buffer);
			} else {
				throw newClosedChannelException();
			}																																															
		}
		
		
		public void write(ByteBuffer buffer, IWriteCompletionHandler writeCompletionHandler) throws IOException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().write(buffer, writeCompletionHandler);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		
		public void write(ByteBuffer[] buffers, IWriteCompletionHandler writeCompletionHandler) throws IOException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().write(buffers, writeCompletionHandler);
			} else {
				throw newClosedChannelException();
			}																																															
		}
		

		public long write(ByteBuffer[] buffers) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(buffers);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public long write(ByteBuffer[] srcs, int offset, int length) throws IOException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(srcs, offset, length);
			} else {
				throw newClosedChannelException();
			}																																																			
		}
		
		public void write(ByteBuffer[] srcs, int offset, int length, IWriteCompletionHandler writeCompletionHandler) throws IOException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().write(srcs, offset, length, writeCompletionHandler);
			} else {
				throw newClosedChannelException();
			}																																																			
        }

		public int write(double d) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(d);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public int write(int i) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(i);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public long write(List<ByteBuffer> buffers) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(buffers);
			} else {
				throw newClosedChannelException();
			}																																																			
		}
		
		
		public void write(List<ByteBuffer> buffers, IWriteCompletionHandler writeCompletionHandler) throws IOException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().write(buffers, writeCompletionHandler);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public int write(long l) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(l);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public int write(short s) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(s);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public int write(String message) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(message);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public int write(String message, String encoding) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().write(message, encoding);
			} else {
				throw newClosedChannelException();
			}																																																			
		}
		
		
		public void write(String message, String encoding, IWriteCompletionHandler writeCompletionHandler) throws IOException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().write(message, encoding, writeCompletionHandler);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public void flush() throws ClosedChannelException, IOException, SocketTimeoutException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().flush();
			} else {
				throw newClosedChannelException();
			}																																																		
		}

		public long transferFrom(FileChannel source) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().transferFrom(source);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public long transferFrom(ReadableByteChannel source) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().transferFrom(source);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public long transferFrom(ReadableByteChannel source, int chunkSize) throws IOException, BufferOverflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().transferFrom(source, chunkSize);
			} else {
				throw newClosedChannelException();
			}																																																			
		}

		public int getPendingWriteDataSize() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().getPendingWriteDataSize();
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}

		public void markWritePosition() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().markWritePosition();
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}

		public void removeWriteMark() {
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().removeWriteMark();
			}																																																		
		}

		public boolean resetToWriteMark() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().resetToWriteMark();
			} else {
				throw newClosedChannelRuntimeException();
			}																																																							
		}

		public void markReadPosition() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}
		
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().markReadPosition();
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}

		public void removeReadMark() {
			if (isDetached) {
				return;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().removeReadMark();
			} 																																																		
		}

		public boolean resetToReadMark() {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				return holder.getConnection().resetToReadMark();
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}
		
	
		public int available() throws IOException {
			if (isDetached) {
				return -1;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().available();
			} else {
				return -1;
			}
		}

				
		public int getReadBufferVersion() throws IOException {
			if (isDetached) {
				return -1;
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().getReadBufferVersion();
			} else {
				return -1;
			}
		}

		public int indexOf(String str) throws IOException {
			if (isDetached) {
				return -1;
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().indexOf(str);
			} else {
				return -1;
			}
		}

		public int indexOf(String str, String encoding) throws IOException {
			if (isDetached) {
				return -1;
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().indexOf(str, encoding);
			} else {
				return -1;
			}
		}
		
		
		public void unread(ByteBuffer[] buffers) throws IOException {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().unread(buffers);
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}
		
		
		public void unread(byte[] bytes) throws IOException {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().unread(bytes);
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}

		

		public void unread(ByteBuffer buffer) throws IOException {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}

			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().unread(buffer);
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}

		
		public void unread(String text) throws IOException {
			if (isDetached) {
				throw newClosedChannelRuntimeException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (isOpen && (holder != null)) {
				holder.getConnection().unread(text);
			} else {
				throw newClosedChannelRuntimeException();
			}																																																			
		}
		
		
		public int read(ByteBuffer buffer) throws IOException {
			if (isDetached) {
				return -1;
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().read(buffer);
			} else {
				return -1;
			}
		}

		public byte readByte() throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readByte();
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public ByteBuffer[] readByteBufferByDelimiter(String delimiter) throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readByteBufferByDelimiter(delimiter);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public ByteBuffer[] readByteBufferByDelimiter(String delimiter, int maxLength) throws IOException, BufferUnderflowException, MaxReadSizeExceededException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readByteBufferByDelimiter(delimiter, maxLength);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public ByteBuffer[] readByteBufferByDelimiter(String delimiter, String encoding) throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readByteBufferByDelimiter(delimiter, encoding);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public ByteBuffer[] readByteBufferByDelimiter(String delimiter, String encoding, int maxLength) throws IOException, BufferUnderflowException, MaxReadSizeExceededException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readByteBufferByDelimiter(delimiter, encoding, maxLength);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public ByteBuffer[] readByteBufferByLength(int length) throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readByteBufferByLength(length);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public byte[] readBytesByDelimiter(String delimiter) throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readBytesByDelimiter(delimiter);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public byte[] readBytesByDelimiter(String delimiter, int maxLength) throws IOException, BufferUnderflowException, MaxReadSizeExceededException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readBytesByDelimiter(delimiter, maxLength);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public byte[] readBytesByDelimiter(String delimiter, String encoding) throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readBytesByDelimiter(delimiter, encoding);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public byte[] readBytesByDelimiter(String delimiter, String encoding, int maxLength) throws IOException, BufferUnderflowException, MaxReadSizeExceededException {
			if (isDetached) {
				throw newClosedChannelException();
			}		
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readBytesByDelimiter(delimiter, encoding, maxLength);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public byte[] readBytesByLength(int length) throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readBytesByLength(length);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public double readDouble() throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readDouble();
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public int readInt() throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
			    return holder.getConnection().readInt();
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public long readLong() throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readLong();
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public short readShort() throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readShort();
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public String readStringByDelimiter(String delimiter) throws IOException, BufferUnderflowException, UnsupportedEncodingException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readStringByDelimiter(delimiter);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public String readStringByDelimiter(String delimiter, int maxLength) throws IOException, BufferUnderflowException, UnsupportedEncodingException, MaxReadSizeExceededException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readStringByDelimiter(delimiter, maxLength);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public String readStringByDelimiter(String delimiter, String encoding) throws IOException, BufferUnderflowException, UnsupportedEncodingException, MaxReadSizeExceededException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readStringByDelimiter(delimiter, encoding);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public String readStringByDelimiter(String delimiter, String encoding, int maxLength) throws IOException, BufferUnderflowException, UnsupportedEncodingException, MaxReadSizeExceededException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readStringByDelimiter(delimiter, encoding, maxLength);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public String readStringByLength(int length) throws IOException, BufferUnderflowException, UnsupportedEncodingException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readStringByLength(length);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public String readStringByLength(int length, String encoding) throws IOException, BufferUnderflowException, UnsupportedEncodingException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().readStringByLength(length, encoding);
			} else {
				throw newClosedChannelException();
			}																																															
		}

		public long transferTo(WritableByteChannel target, int length) throws IOException, BufferUnderflowException {
			if (isDetached) {
				throw newClosedChannelException();
			}
			
			NativeConnectionHolder holder = nativeConnectionHolder;
			if (holder != null) {
				return holder.getConnection().transferTo(target, length);
			} else {
				throw newClosedChannelException();
			}																																															
		}
		
		private ExtendedClosedChannelException newClosedChannelException() {
			return new ExtendedClosedChannelException("channel " + getId() + " is already closed");
		}
		
		private RuntimeException newClosedChannelRuntimeException() {
			return new RuntimeException("channel " + getId() + " is already closed");
		}
	}
}
