/*
* Copyright 2022 Yak.Works - Licensed under the Apache License, Version 2.0 (the "License")
* You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
*/
package yakworks.security.rest.okta

import groovy.transform.CompileStatic

import org.opensaml.saml.common.xml.SAMLConstants
import org.pac4j.core.http.callback.NoParameterCallbackUrlResolver
import org.pac4j.core.util.CommonHelper
import org.pac4j.saml.client.SAML2Client
import org.pac4j.saml.config.SAML2Configuration
import org.springframework.core.io.Resource
import org.springframework.core.io.UrlResource
import org.springframework.core.io.WritableResource

import yakworks.commons.lang.Validate

@CompileStatic
class OktaSAML2Client extends SAML2Client {

    String serverUrl

    OktaSAML2Client() {
        super(new SAML2Configuration())
        //the default is QueryParameterCallbackUrlResolver : Which adds one extra parameter ?client_name=OktaSAML2Client
        //the same url is used in the spMetaData xml generated by the pac4j.
        //We do not need client name in there because rest security takes care or if, and we have no other parameteters to be passed.
        //this enables us to have simple url like `api/oauth/saml/sso`
        callbackUrlResolver = new NoParameterCallbackUrlResolver()
    }

    OktaSAML2Client(final SAML2Configuration configuration) {
        super(configuration)
        callbackUrlResolver = new NoParameterCallbackUrlResolver()
    }

    void setKeystore(Resource r) {
        configuration.setKeystoreResource(r)
    }

    void setKeystorePassword(String pwd) {
        configuration.setKeystorePassword(pwd)
    }

    void setKeyPassword(String pwd) {
        configuration.setPrivateKeyPassword(pwd)
    }

    /**
     * The identity provider / okta metadata xml locatopn
     */
    void setIdpMetadata(Resource r) {
        configuration.setIdentityProviderMetadataResource(r)
    }

    void setIdpMetadataUrl(String path) {
        setIdpMetadata(new UrlResource(path))
    }

    /*
     * Service provider metadata location - the location needs to be writable.
     * This is where Pac4J will generate SP metadata file.
     */
    void setSpMetadata(WritableResource r) {
        configuration.setServiceProviderMetadataResource(r);
    }

    /*
     * Used to generate Service provider entity Id.
     */
    void setServerUrl(String s) {
        Validate.notEmpty(s)
        this.serverUrl = s
    }

    @Override
    protected void clientInit() {
        CommonHelper.assertNotNull("configuration", getConfiguration())
        configuration.setAuthnRequestBindingType(SAMLConstants.SAML2_REDIRECT_BINDING_URI);
        configuration.setServiceProviderEntityId("${serverUrl}/api/oauth/saml/sso")
        setCallbackUrl("${serverUrl}/api/oauth/saml/sso")
        super.clientInit()
    }

}
