/*
 * Copyright LGPL3
 * YES Technology Association
 * http://yestech.org
 *
 * http://www.opensource.org/licenses/lgpl-3.0.html
 */

/*
 *
 * Author:  Artie Copeland
 * Last Modified Date: $DateTime: $
 */
package org.yestech.lib.web;

import static org.yestech.lib.util.LoggingUtils.logRequestHeaders;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.apache.commons.lang.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.Enumeration;

/**
 * A collection of request processing utilities
 *
 * @author Artie Copeland
 * @version $Revision: $
 */
final public class RequestUtils {
    final private static Logger logger = LoggerFactory.getLogger(RequestUtils.class);
    
    private RequestUtils() {
    }

    /**
     * Checks if the request uri is an include.
     * These cannot be gzipped.
     */
    public static boolean isIncluded(final HttpServletRequest request) {
        final String uri = (String) request.getAttribute("javax.servlet.include.request_uri");
        final boolean includeRequest = !(uri == null);

        if (includeRequest && logger.isDebugEnabled()) {
            logger.debug(request.getRequestURL() + " resulted in an include request. This is unusable, because" +
                    "the response will be assembled into the overrall response. Not gzipping.");
        }
        return includeRequest;
    }

    /**
     * Determine whether the user agent accepts GZIP encoding. This feature is part of HTTP1.1.
     * If a browser accepts GZIP encoding it will advertise this by including in its HTTP header:
     * <p/>
     * <code>
     * Accept-Encoding: gzip
     * </code>
     * <p/>
     * Requests which do not accept GZIP encoding fall into the following categories:
     * <ul>
     * <li>Old browsers, notably IE 5 on Macintosh.
     * <li>Internet Explorer through a proxy. By default HTTP1.1 is enabled but disabled when going
     * through a proxy. 90% of non gzip requests seen on the Internet are caused by this.
     * </ul>
     * As of September 2004, about 34% of Internet requests do not accept GZIP encoding.
     *
     * @param request
     * @return true, if the User Agent request accepts GZIP encoding
     */
    public static boolean acceptsGzipEncoding(HttpServletRequest request) {
        return acceptsEncoding(request, "gzip");
    }

    /**
     * Checks if request accepts the named encoding.
     */
    public static boolean acceptsEncoding(final HttpServletRequest request, final String name) {
        final boolean accepts = headerContains(request, "Accept-Encoding", name);
        return accepts;
    }

    /**
     * Checks if request contains the header value.
     *
     * @return true if header acc
     */
    public static boolean headerContains(final HttpServletRequest request, final String header, final String value) {

        logRequestHeaders(logger, request);

        final Enumeration accepted = request.getHeaders(header);
        while (accepted.hasMoreElements()) {
            final String headerValue = (String) accepted.nextElement();
            if (headerValue.indexOf(value) != -1) {
                return true;
            }
        }
        return false;
    }

    /**
     * Trys to resolve an clients ip address first by last item in "X-Forwarded-For" HTTP header.
     * If that fails then use {@link javax.servlet.http.HttpServletRequest#getRemoteAddr()}.
     *  
     * @param request HttpRequest
     * @return Ip Address
     */
    public static String resolveUserIpAddress(HttpServletRequest request) {
        String endUserIp = request.getHeader("X-Forwarded-For");
        String[] ips = StringUtils.split(endUserIp, ",");
        if (ips != null && ips.length > 0) {
            endUserIp = ips[ips.length - 1];
        } else {
            endUserIp = request.getRemoteAddr();
        }
        return endUserIp.trim();
    }
}
