package org.yetiz.utils.hbase;

import org.apache.hadoop.hbase.client.*;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.yetiz.utils.hbase.exception.UnHandledException;
import org.yetiz.utils.hbase.exception.YHBaseException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Created by yeti on 16/4/5.
 */
public class HBaseTable {
	private final HTableInterface table;
	private final TableName tableName;
	private final Model model;
	private boolean closed = false;

	protected HBaseTable(TableName tableName,
	                     HTableInterface table) {
		this.tableName = tableName;
		this.table = table;
		this.model = new Model();
	}

	public <R extends HTableModel> Model<R> model() {
		return model;
	}

	public void close() {
		try {
			table().close();
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		} finally {
			closed = true;
		}
	}

	private HTableInterface table() {
		return table;
	}

	private YHBaseException convertedException(Throwable throwable) {
		if (throwable instanceof YHBaseException) {
			return (YHBaseException) throwable;
		} else {
			return new UnHandledException(throwable);
		}
	}

	public boolean isClosed() {
		return closed;
	}

	public boolean exists(Get get) {
		try {
			return table().exists(get);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public boolean[] exists(List<Get> gets) {
		try {
			Boolean[] tmp = table().exists(gets);
			boolean[] rtn = new boolean[tmp.length];
			for (int i = 0; i < tmp.length; i++) {
				rtn[i] = tmp[i];
			}
			return rtn;
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public Result append(Append append) {
		try {
			return table().append(append);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public TableName tableName() {
		return tableName;
	}

	public Result increment(Increment increment) {
		try {
			return table().increment(increment);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public Result get(Get get) {
		try {
			return table().get(get);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public Result[] get(List<Get> gets) {
		try {
			return table().get(gets);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public void put(Put put) {
		try {
			table().put(put);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public void put(List<Put> puts) {
		try {
			table().put(puts);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public void delete(Delete delete) {
		try {
			table().delete(delete);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public void delete(List<Delete> deletes) {
		try {
			table().delete(deletes);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public ResultScanner scan(Scan scan) {
		try {
			return table().getScanner(scan);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public void batch(List<? extends Row> actions, Object[] results) {
		try {
			table().batch(actions, results);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public <R> void batchCallback(List<? extends Row> actions,
	                              Object[] results,
	                              Batch.Callback<R> callback) {
		try {
			table().batchCallback(actions, results, callback);
		} catch (Throwable throwable) {
			throw convertedException(throwable);
		}
	}

	public class Model<R extends HTableModel> {

		public R append(Append append) {
			try {
				return convert(table().append(append));
			} catch (Throwable throwable) {
				throw convertedException(throwable);
			}
		}

		private R convert(Result result) {
			return HTableModel.newWrappedModel(tableName, result);
		}

		public R increment(Increment increment) {
			try {
				return convert(table().increment(increment));
			} catch (Throwable throwable) {
				throw convertedException(throwable);
			}
		}

		public R get(Get get) {
			try {
				return convert(table().get(get));
			} catch (Throwable throwable) {
				throw convertedException(throwable);
			}
		}

		public List<R> get(List<Get> gets) {
			try {
				ArrayList<R> rs = new ArrayList<>();

				for (Result result : Arrays.asList(table().get(gets))) {
					rs.add(convert(result));
				}

				return rs;
			} catch (Throwable throwable) {
				throw convertedException(throwable);
			}
		}

		public ReturnScanner<R> scan(Scan scan) {
			try {
				return new ReturnScanner(table().getScanner(scan));
			} catch (Throwable throwable) {
				throw convertedException(throwable);
			}
		}

		public class ReturnScanner<R extends HTableModel> {
			private ResultScanner scanner;

			public ReturnScanner(ResultScanner scanner) {
				this.scanner = scanner;
			}

			public R next() throws IOException {
				return convert(scanner.next());
			}

			private R convert(Result result) {
				return HTableModel.newWrappedModel(tableName, result);
			}

			public List<R> next(int nbRows) throws IOException {
				List<R> rs = new ArrayList<>();
				for (Result result : Arrays.asList(scanner.next(nbRows))) {
					rs.add(convert(result));
				}

				return rs;
			}

			public void close() {
				scanner.close();
			}
		}
	}
}
