package org.yggd.client.sftp;

import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;

import java.io.File;
import java.net.URISyntaxException;
import java.util.Properties;

public class SftpClientBuilder {

    private final String host;
    private final int port;
    private final JSch jSch = new JSch();

    private String username;

    private Properties properties = new Properties();

    private File privateKey;

    private SftpClient client;

    public SftpClientBuilder(String host, int port) {
        this.host = host;
        this.port = port;
        properties.put("StrictHostKeyChecking", "no");
    }

    public SftpClientBuilder username(String username) {
        this.username = username;
        return this;
    }

    public SftpClientBuilder property(Object key, Object value) {
        this.properties.put(key, value);
        return this;
    }

    public SftpClientBuilder privateKey(String privateKey) {
        try {
            jSch.addIdentity(resourceToFile(privateKey).getAbsolutePath());
        } catch (JSchException e) {
            throw new IllegalArgumentException(e);
        }
        return this;
    }

    public SftpClientBuilder privateKey(String privateKey, String passphrase) {
        try {
            jSch.addIdentity(resourceToFile(privateKey).getAbsolutePath(), passphrase.getBytes());
        } catch (JSchException e) {
            throw new IllegalArgumentException(e);
        }
        return this;
    }

    public SftpClientImpl build() {
        final Session session = createSession();
        assert session != null;
        return new SftpClientImpl(session, createChannel(session));
    }

    private Session createSession() {
        try {
            Session session = jSch.getSession(username, host, port);
            session.setConfig(properties);
            session.connect();
            return session;
        } catch (JSchException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private ChannelSftp createChannel(Session session) {
        try {
            ChannelSftp channelSftp = (ChannelSftp) session.openChannel("sftp");
            channelSftp.connect();
            return channelSftp;
        } catch (JSchException e) {
            throw new IllegalStateException(e);
        }
    }

    private File resourceToFile(String resource) {
        try {
            return new File(SftpClientBuilder.class.getResource(resource).toURI());
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException(e);
        }
    }
}
