/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2022
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.api.core.runnable;

import org.gradle.api.Action;
import org.gradle.api.DefaultTask;
import org.gradle.api.tasks.Internal;
import org.gradle.api.tasks.Nested;
import org.gradle.api.tasks.TaskAction;
import org.gradle.process.ExecResult;
import org.ysb33r.grolifant.api.core.CmdlineArgumentSpec;
import org.ysb33r.grolifant.api.core.executable.ProcessExecutionSpec;
import org.ysb33r.grolifant.api.core.ProjectOperations;
import org.ysb33r.grolifant.api.core.jvm.JavaForkOptionsWithEnvProvider;
import org.ysb33r.grolifant.api.core.jvm.JvmEntryPoint;

/**
 * Configures a task that will run an execution in a forked JVM.
 *
 * @author Schalk W. Cronj̧é
 * @since 2.0
 */
public class AbstractJvmExecTask<T extends AbstractJvmExecSpec> extends DefaultTask implements ForkedJvmExecutable<T> {
    /**
     * Configures a JVM that will be forked.
     *
     * @param configurator An action to configure the JVM's fork options.
     * @return The configured executable
     */
    @Override
    public T jvm(Action<JavaForkOptionsWithEnvProvider> configurator) {
        return (T)getJvmExecSpec().jvm(configurator);
    }

    /**
     * Configures the arguments.
     *
     * @param configurator An action to configure the arguments.
     * @return The configured executable
     */
    @Override
    public T runnerSpec(Action<CmdlineArgumentSpec> configurator) {
        return ((T) (getJvmExecSpec().runnerSpec(configurator)));
    }

    /**
     * Configures the entrypoint for the JVM executable
     *
     * @param configurator An action to configure the entrypoint
     * @return The configured executable.
     */
    @Override
    public T entrypoint(Action<JvmEntryPoint> configurator) {
        return ((T) (getJvmExecSpec().entrypoint(configurator)));
    }

    /**
     * Configures the stream redirection and exit code checks.
     *
     * @param configurator An action to configure the execution.
     * @return The configured executable.
     */
    @Override
    public T process(Action<ProcessExecutionSpec> configurator) {
        return ((T) (getJvmExecSpec().process(configurator)));
    }

    protected AbstractJvmExecTask() {
        this.projectOperations = ProjectOperations.find(getProject());
    }

    @Nested
    protected T getJvmExecSpec() {
        return this.execSpec;
    }

    protected void setExecSpec(T execSpec) {
        this.execSpec = execSpec;
    }

    @Internal
    protected ProjectOperations getProjectOperations() {
        return this.projectOperations;
    }

    @TaskAction
    protected void exec() {
        ExecResult result = projectOperations.javaexec( x -> getJvmExecSpec().copyTo(x));
        result.assertNormalExitValue();
    }

    private T execSpec;
    private final ProjectOperations projectOperations;
}
