/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2022
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.core

import groovy.transform.CompileStatic
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ConfigurationContainer
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.ProjectTools
import org.ysb33r.grolifant.api.core.StringTools
import org.ysb33r.grolifant.internal.core.IterableUtils
import org.ysb33r.grolifant.internal.core.Transform

import java.util.function.Function

import static org.ysb33r.grolifant.internal.core.IterableUtils.treatAsIterable

/**
 *
 * @author Schalk W. Cronje
 *
 * @since 2.0
 */
@SuppressWarnings('AbstractClassWithoutAbstractMethod')
@CompileStatic
abstract class ProjectToolsProxy implements ProjectTools {
    /**
     * Resolves an arbitrary item to a {@link Configuration} instance.
     *
     * @param configurationThingy Instance or {@link Configuration} or something that resolves to a string.
     * @return Configuration
     */
    @Override
    Configuration asConfiguration(Object configurationThingy) {
        Transform.convertItem(
            configurationThingy,
            transformer
        )
    }

    /**
     * Resolves arbitrary items to a collection of {@link Configuration} instances.
     *
     * @param configurationThingies Collection that might contain {@link Configuration} or string-type instances
     * @return Collection of resolved {@link Configuration} instances
     */
    @Override
    Collection<Configuration> asConfigurations(Collection<?> configurationThingies) {
        Set<Configuration> resolvedConfigurations = []
        configurationThingies.forEach { thingy ->
            switch (thingy) {
                case Configuration:
                    resolvedConfigurations.add((Configuration) thingy)
                    break
                case Collection:
                    resolvedConfigurations.addAll(asConfigurations((Collection) thingy))
                    break
                default:
                    if (treatAsIterable(thingy)) {
                        final asList = IterableUtils.toList((Iterable) thingy)
                        resolvedConfigurations.addAll(asConfigurations((Collection) asList))
                    } else {
                        resolvedConfigurations.add(asConfiguration(thingy))
                    }
            }
        }
        resolvedConfigurations
    }

    protected ProjectToolsProxy(ProjectOperations po, Project tempProjectReference) {
        this.projectOperations = po
        this.configurations = tempProjectReference.configurations
    }

    protected final ProjectOperations projectOperations

    private Function<Object, Configuration> getTransformer() {
        CONFIGURATION_TRANSFORMER.curry(
            configurations,
            projectOperations.stringTools
        ) as Function<Object, Configuration>
    }

    private final ConfigurationContainer configurations

    private static final Closure CONFIGURATION_TRANSFORMER = {
        ConfigurationContainer c, StringTools st, Object configurationThingy ->
            switch (configurationThingy) {
                case Configuration:
                    (Configuration) configurationThingy
                    break
                default:
                    c.getByName(st.stringize(configurationThingy))
            }
    }
}
