/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2022
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.core

import groovy.transform.CompileStatic
import org.gradle.api.GradleException
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.file.FileCollection
import org.gradle.api.provider.Provider
import org.gradle.api.provider.ProviderFactory
import org.gradle.api.tasks.TaskContainer
import org.gradle.api.tasks.TaskInputFilePropertyBuilder
import org.gradle.api.tasks.TaskInputs
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.StringTools
import org.ysb33r.grolifant.api.core.TaskTools
import org.ysb33r.grolifant.api.errors.UnexpectedNullException
import org.ysb33r.grolifant.api.errors.UnsupportedConfigurationException
import org.ysb33r.grolifant.internal.core.Transform

import java.util.function.Function

/**
 * Non-Gradle API-specific common code from {@link TaskTools}.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
abstract class TaskToolsProxy implements TaskTools {
    /**
     * This is similar to putting both a {@link org.gradle.api.tasks.SkipWhenEmpty} and a
     * {@code org.gradle.api.tasks.IgnoreEmptyDirectories} annotation on a property, but it deals with both
     * backwards and forward compatibility around Gradle 6.8.
     * <p>
     * Accepts any of the following
     *     <ul>
     *         <li>{@link FileCollection}</li>
     *         <li>{@link org.gradle.api.file.Directory}</li>
     *         <li>File as long as it points to a directory</li>
     *         <li>ANy Provider, Supplier, Optional, Callable</li>
     *     </ul>
     * </p>
     *
     * @param inputs Inputs for a specific task
     * @param files File collections
     */
    @Override
    void ignoreEmptyDirectories(TaskInputs inputs, Object files) {
        switch (files) {
            case FileCollection:
                markIgnoreEmptyDirectories(inputs.files(files))
                break
            default:
                markIgnoreEmptyDirectories(
                    inputs.files { ->
                        projectOperations.fsOperations.filesDropNull([files])
                    }
                )
        }
    }

    /**
     * Resolves a list of many objects to tasks items to a provider of tasks.
     *
     * @param taskies Things that can be converted to {@link Task} or {@link org.gradle.api.tasks.TaskProvider}
     * @return A provider to the list of resolved tasks.
     */
    @Override
    Provider<List<? extends Task>> taskize(Object... taskies) {
        taskize(taskies as List)
    }

    /**
     * Resolves a list of many objects to tasks items to a provider of tasks.
     *
     * @param taskies Things that can be converted to {@link org.gradle.api.Task}
     * or {@link org.gradle.api.tasks.TaskProvider}
     * @return A provider to the list of resolved tasks.
     */
    @Override
    Provider<List<? extends Task>> taskize(Iterable<Object> taskies) {
        final transformer = singleTransformer(this.tasks)
        providerFactory.provider { ->
            List<? extends Task> resolvedTasks = []
            Transform.convertItems(taskies as List, resolvedTasks, transformer)
            resolvedTasks
        }
    }

    /**
     * Creates a basic task {@link Provider} instance from an object.
     *
     * @param tasks Override the task containerto use.
     * @param pf Override the provider factory to use.
     * @param o Object to be evaluated to a task.
     * @return Lazy-evaluatable task.
     * @since 2.0
     */
    @Override
    Provider<? extends Task> taskProviderFrom(TaskContainer tasks1, ProviderFactory pf, Object o) {
        if (isTaskProvider(o)) {
            (Provider<Task>) o
        } else {
            final transformer = singleTransformer(tasks1)

            pf.provider { ->
                Transform.convertItem(o, transformer)
            }
        }
    }

    /**
     * Creates a basic task {@link Provider} instance from an object.
     *
     * @param o Object to be evaluated to a task.
     * @return Lazy-evaluatable task.
     * @since 2.0
     */
    @Override
    Provider<? extends Task> taskProviderFrom(Object o) {
        taskProviderFrom(tasks, providerFactory, o)
    }

    protected TaskToolsProxy(ProjectOperations incompleteReference, Project project) {
        this.projectOperations = incompleteReference
        this.providerFactory = project.providers
        this.tasks = project.tasks
    }

    protected final ProjectOperations projectOperations
    protected final ProviderFactory providerFactory
    protected final TaskContainer tasks

    /**
     * Whether this is a {@code TaskProvider<?>}.
     *
     * @param o Object to evaluate
     * @return {@code true} is an instance of {@link org.gradle.api.tasks.TaskProvider}.
     */
    abstract protected boolean isTaskProvider(Object o)

    /**
     * Marks an input property to ignore empty directories.
     *
     * @param inputsBuilder Property builder to update
     */
    abstract protected void markIgnoreEmptyDirectories(TaskInputFilePropertyBuilder inputsBuilder)

    private Function<Object, Task> singleTransformer(TaskContainer t1) {
        { TaskContainer tasks1, StringTools st, Object t ->
            try {
                switch (t) {
                    case null:
                        throw new UnexpectedNullException('Null values are not allowed')
                    case Task:
                        (Task) t
                        break
                    default:
                        tasks1.getByName(st.stringize(t))
                }
            } catch (final GradleException e) {
                throw new UnsupportedConfigurationException("Item '${t}' cannot be resolved to a task", e)
            }
        }.curry(t1, projectOperations.stringTools) as Function<Object, Task>
    }
}
