/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2022
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.api.core.executable

import groovy.transform.CompileStatic
import org.gradle.api.provider.Provider
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.Internal
import org.ysb33r.grolifant.api.core.CmdlineArgumentSpec
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.runnable.AbstractCmdlineArgumentSpec

import java.util.concurrent.Callable

/**
 * Defines a base class for implementing a script definition.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class ScriptDefinition extends AbstractCmdlineArgumentSpec implements CmdlineArgumentSpec, ScriptSpec {

    ScriptDefinition(ProjectOperations po) {
        super(po.stringTools, po.providers)
        nameProvider = po.provider(new Callable<String>() {
            @Override
            String call() {
                po.stringTools.stringizeOrNull(nameStore.empty ? null : nameStore[0])
            }
        })
        pathProvider = po.provider(new Callable<File>() {
            @Override
            File call() {
                pathStore.empty ? null : po.fsOperations.file(pathStore[0])
            }
        })
        nameOrPathProvider = po.providerTools.orElse(
            nameProvider,
            pathProvider.map { it?.absolutePath }
        )
        preArgsProvider = nameOrPathProvider.map { [it] }
    }

    void setName(Object lazyName) {
        this.nameStore.clear()
        this.pathStore.clear()
        this.nameStore.add(lazyName)
    }

    void setPath(Object lazyPath) {
        this.nameStore.clear()
        this.pathStore.clear()
        this.pathStore.add(lazyPath)
    }

    @Internal
    Provider<String> getName() {
        nameProvider
    }

    @Internal
    Provider<File> getPath() {
        pathProvider
    }

    @Input
    Provider<String> getNameOrPath() {
        nameOrPathProvider
    }

    /**
     * A provider to arguments that will be inserted before any supplied arguments.
     *
     * @return Arguments provider
     */
    @Override
    Provider<List<String>> getPreArgs() {
        this.preArgsProvider
    }

    private final List<Object> nameStore = []
    private final List<Object> pathStore = []
    private final Provider<String> nameProvider
    private final Provider<File> pathProvider
    private final Provider<String> nameOrPathProvider
    private final Provider<List<String>> preArgsProvider
}
