/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.api.core.runnable;

import org.gradle.api.Action;
import org.gradle.api.tasks.Nested;
import org.gradle.process.JavaExecSpec;
import org.ysb33r.grolifant.api.core.ProjectOperations;
import org.ysb33r.grolifant.api.core.executable.ScriptSpec;
import org.ysb33r.grolifant.api.core.executable.ScriptDefinition;

/**
 * Defines the basics for executing a script from a JVM-based scripting language.
 *
 * @since 2.0
 */
abstract public class AbstractJvmScriptExecSpec<T extends AbstractJvmScriptExecSpec<T>>
        extends AbstractJvmExecSpec<T>
        implements ForkedJvmScript<T> {

    /**
     * Configures the script specification.
     *
     * @param configurator Action to configure the script.
     * @return This object.
     */
    @Override
    public T script(Action<ScriptSpec> configurator) {
        configurator.execute(getScriptDefinition());
        return (T) this;
    }

    /**
     * Copies this specification to a standard {@link JavaExecSpec}.
     * Adds in the script and all of the script arguments.
     *
     * @param execSpec Target execution specification.
     */
    @Override
    public void copyTo(JavaExecSpec execSpec) {
        super.copyTo(execSpec);
        copyScriptDetailsToExecSpec(execSpec);
    }

    protected AbstractJvmScriptExecSpec(ProjectOperations po) {
        super(po);
        this.jvmScript = new ScriptDefinition(po);
    }

    /**
     * Returns the instance of the script definition.
     *
     * @return Script definition object. Never {@code null}.
     */
    @Nested
    protected ScriptSpec getScriptDefinition() {
        return this.jvmScript;
    }

    /**
     * Copies the script arguments to the execution specification
     * The default behaviour is to copy the script name and then the script arguments.
     */
    protected void copyScriptDetailsToExecSpec(JavaExecSpec execSpec) {
        execSpec.args(jvmScript.getNameOrPath().get());
        execSpec.args(jvmScript.getAllArgs().get());
    }

    private final ScriptSpec jvmScript;
}
