/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.core

import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import org.gradle.api.Project
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.CopySpec
import org.gradle.api.file.FileTree
import org.gradle.api.internal.file.copy.CopySpecInternal
import org.gradle.api.provider.Property
import org.gradle.api.provider.ProviderFactory
import org.ysb33r.grolifant.api.core.FileSystemOperations
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.internal.core.TempDirectory
import org.ysb33r.grolifant.internal.core.Transform

import java.nio.file.Path
import java.nio.file.Paths
import java.util.regex.Pattern

import static org.ysb33r.grolifant.internal.core.Transform.convertItems
import static org.ysb33r.grolifant.internal.core.Transform.convertItemsDropNull

/**
 * Common filesystem operations
 *
 * @author Schalk W. Cronjé
 *
 * @since 1.3
 */
@SuppressWarnings('AbstractClassWithoutAbstractMethod')
@CompileStatic
abstract class FileSystemOperationsProxy implements FileSystemOperations {

    /**
     * Creates a temporary directory with the given prefix.
     *
     * Directory will be set to delete on VM exit, but consumers are allowed to delete earlier.
     *
     * The directory will be created somewhere in the build directory.
     *
     * @param prefix Prefix for directory.
     * @return Directory
     *
     * @since 2.0
     */
    @Override
    File createTempDirectory(String prefix) {
        tmpDirFactory.createTempDirectory(prefix)
    }

    /**
     * Converts a file-like objects to {@link java.io.File} instances with project context.
     * <p>
     * Converts collections of the following recursively until it gets to a file:
     *
     * <ul>
     *   <li> {@code CharSequence} including {@code String} and {@code GString}.
     *   <li> {@link java.io.File}.
     *   <li> {@link java.nio.file.Path} is it is associated with the default provider
     *   <li> URLs and URis of {@code file:} schemes.
     *   <li> Groovy Closures.
     *   <li> {@link java.util.concurrent.Callable}.
     *   <li> {@link org.gradle.api.provider.Provider}.
     *   <li> {@link org.gradle.api.file.Directory}
     *   <li> {@link org.gradle.api.resources.TextResource}
     * </ul>
     *
     * Collections are flattened.
     * Null instances are not allowed.
     *
     * @param files Potential {@link File} objects
     * @return File collection.
     */
    @Override
    ConfigurableFileCollection files(Collection<?> files) {
        List<File> outputs = []
        emptyFileCollection().from(convertItems(files.toList(), outputs) { f -> file(f) })
    }

    /**
     * Converts a file-like objects to {@link java.io.File} instances with project context.
     * <p>
     * Converts collections of the following recursively until it gets to a file:
     *
     * <ul>
     *   <li> {@code CharSequence} including {@code String} and {@code GString}.
     *   <li> {@link java.io.File}.
     *   <li> {@link java.nio.file.Path} is it is associated with the default provider
     *   <li> URLs and URis of {@code file:} schemes.
     *   <li> Groovy Closures.
     *   <li> {@link java.util.concurrent.Callable}.
     *   <li> {@link org.gradle.api.provider.Provider}.
     *   <li> {@link org.gradle.api.file.Directory}
     *   <li> {@link org.gradle.api.resources.TextResource}
     * </ul>
     *
     * Collections are flattened.
     * Null instances are removed.
     *
     * @param files Potential {@link File} objects
     * @return File collection.
     */
    @Override
    ConfigurableFileCollection filesDropNull(Collection<?> files) {
        List<File> outputs = []
        emptyFileCollection().from(convertItemsDropNull(files.toList(), outputs) { f -> file(f) })
    }

    /** Provides a list of directories below another directory
     *
     * @param distDir Directory
     * @return List of directories. Can be empty if, but never {@code null}
     *   supplied directory.
     */
    @Override
    List<File> listDirs(File distDir) {
        if (distDir.exists()) {
            distDir.listFiles(new FileFilter() {
                @Override
                boolean accept(File pathname) {
                    pathname.directory
                }
            }) as List<File>
        } else {
            []
        }
    }

    /**
     * Returns the relative path from the project directory to the given path.
     *
     * @param f Object that is resolvable to a file within project context.
     * @return Relative path. Never {@code null}.
     */
    @Override
    String relativePath(Object f) {
        relativizeImpl(projectDirPath, file(f).toPath())
    }

    /**
     * Returns the relative path from the root project directory to the given path.
     *
     * @param f Object that is resolvable to a file within project context
     * @return Relative path. Never {@code null}.
     *
     */
    @Override
    String relativeRootPath(Object f) {
        relativizeImpl(rootDirPath, file(f).toPath())
    }

    /**
     * Returns the relative path from the given path to the project directory.
     *
     * @param f Object that is resolvable to a file within project context
     * @return Relative path. Never {@code null}.
     */
    @Override
    String relativePathToProjectDir(Object f) {
        relativizeImpl(file(f).toPath(), projectDirPath)
    }

    /**
     * Returns the relative path from the given path to the root project directory.
     *
     * @param f Object that is resolvable to a file within project context
     * @return Relative path. Never {@code null}.
     */
    @Override
    String relativePathToRootDir(Object f) {
        relativizeImpl(file(f).toPath(), rootDirPath)
    }

    /**
     * Given a {@Link CopySpec}, resolve all of the input files in to a collection.
     *
     * @param copySpec Input specification
     * @return Colelction of files.
     */
    @Override
    FileTree resolveFilesFromCopySpec(CopySpec copySpec) {
        ((CopySpecInternal) copySpec).buildRootResolver().allSource
    }

    /** Converts a string into a string that is safe to use as a file name. T
     *
     * The result will only include ascii characters and numbers, and the "-","_", #, $ and "." characters.
     *
     * @param name A potential file name
     * @return A name that is safe on the local filesystem of the current operating system.
     */
    @Override
    @CompileDynamic
    String toSafeFileName(String name) {
        name.replaceAll(SAFE_FILENAME_REGEX) { String match ->
            String bytes = match.bytes.collect { int it -> Integer.toHexString(it) }.join('')
            "#${bytes}!"
        }
    }

    /**
     * Converts a collection of String into a {@link Path} with all parts guarantee to be safe file parts
     *
     * @param parts File path parts
     * @return File path
     * @since 2.0
     */
    @Override
    Path toSafePath(String... parts) {
        List<String> safeParts = Transform.toList(parts as List) { String it -> toSafeFileName(it) }
        safeParts.size() > 0 ? Paths.get(safeParts[0], safeParts[1..-1].toArray() as String[]) : Paths.get(safeParts[0])
    }

    /**
     * Updates a {@code Property<File>}.
     * <p>
     *     This method deals with a wide range of possibilities and works around the limitations of
     * {@code Property.set()}
     * </p>
     * @param provider Current provider
     * @param fileTarget Value that should be lazy-resolved.
     *
     * @since 2.0
     */
    @Override
    void updateFileProperty(Property<File> provider, Object fileTarget) {
        if (fileTarget == null) {
            provider.set((File) null)
        } else {
            provider.set(providerFactory.provider { ->
                fileOrNull(fileTarget)
            })
        }
    }

    /**
     * The project directory.
     */
    protected final File projectDir

    /**
     * The root project directory.
     */
    protected final File rootDir

    /**
     * The project directory as {@link Path}.
     */
    protected final Path projectDirPath

    /**
     * The project root directory as {@link Path}.
     */
    protected final Path rootDirPath

    /**
     * Provider factory
     */
    protected final ProviderFactory providerFactory

    protected FileSystemOperationsProxy(ProjectOperations incompleteReference, Project tempProjectReference) {
        this.projectDir = tempProjectReference.projectDir
        this.rootDir = tempProjectReference.rootDir
        this.projectDirPath = tempProjectReference.projectDir.toPath()
        this.rootDirPath = tempProjectReference.rootDir.toPath()
        this.tmpDirFactory = new TempDirectory(incompleteReference)
        this.providerFactory = tempProjectReference.providers
    }

    private String relativizeImpl(Path base, Path target) {
        base.relativize(target).toFile().toString()
    }

    private final TempDirectory tmpDirFactory
    private static final Pattern SAFE_FILENAME_REGEX = ~/[^\w_\-.$]/
}