/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.api.core;

import org.gradle.api.Action;
import org.gradle.api.Project;
import org.gradle.process.ExecSpec;
import org.ysb33r.grolifant.api.core.downloader.Downloader;
import org.ysb33r.grolifant.api.core.executable.AppRunnerSpec;
import org.ysb33r.grolifant.api.core.executable.CommandEntryPoint;
import org.ysb33r.grolifant.api.core.executable.ExecutableEntryPoint;
import org.ysb33r.grolifant.internal.core.executable.ExecUtils;
import org.ysb33r.grolifant.internal.core.loaders.ExecToolsLoader;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.util.function.Function;

/**
 * Various tools to deal with non-JVM, out-of-process executables.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
public interface ExecTools {

    static ExecTools load(ProjectOperations incompleteReference, Project project) {
        return ExecToolsLoader.load(incompleteReference, project);
    }

    /**
     * Returns something that looks like an {@link ExecSpec}.
     *
     * @return Instance of an executable specification.
     */
    ExecSpec execSpec();

    /**
     * Creates a {@link AppRunnerSpec}.
     *
     * THis is primarily used internally by classes that implements execution specifications for non-JVM processes.
     *
     * @return Implementation of {@link AppRunnerSpec}
     */
    AppRunnerSpec appRunnerSpec();

    /**
     * Returns an implementation that is optimised for the running version of Gradle.
     *
     * @return Instance of {@link CommandEntryPoint}. Never {@code null}
     */
    CommandEntryPoint commandEntryPoint();

    /**
     * Creates a new downloader for downloading packages / distributions.
     *
     * @param distributionName Name of distribution to be downloaded.
     * @return Instance of {@link Downloader} that is optimised for the running Gradle version.
     */
    Downloader downloader(final String distributionName);

    /**
     * Simplifies running an executable to obtain a version.
     * This is primarily used to implement a {@code runExecutableAndReturnVersion} method.
     *
     * @param argsForVersion Arguments required to obtain a version
     * @param executablePath Location of the executable
     * @param versionParser A parser for the output of running the executable which could extract the version.
     * @return The version string.
     */
    default String parseVersionFromOutput(
            Iterable<String> argsForVersion,
            File executablePath,
            Function<String, String> versionParser
    ) {
        return parseVersionFromOutput(
                argsForVersion,
                executablePath,
                versionParser,
                null
        );
    }

    /**
     * Simplifies running an executable to obtain a version.
     * This is primarily used to implement a {@code runExecutableAndReturnVersion} method.
     *
     * @param argsForVersion Arguments required to obtain a version
     * @param executablePath Location of the executable
     * @param versionParser A parser for the output of running the executable which could extract the version.
     * @param configurator Additional configurator to customise the execution specification.
     * @return The version string.
     */
    String parseVersionFromOutput(
            Iterable<String> argsForVersion,
            File executablePath,
            Function<String, String> versionParser,
            Action<ExecSpec> configurator
    );
}
